/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2018
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.hollow.datamodel.consumer;

import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import net.bluemind.directory.hollow.datamodel.AddressBookRecord;

public class FilteredDirectorySearch extends DefaultDirectorySearch implements BrowsableDirectorySearch {

	private final Predicate<AddressBookRecord> filter;

	public FilteredDirectorySearch(IDirectoryDeserializer deserializer, Predicate<AddressBookRecord> filter) {
		super(deserializer);
		this.filter = filter;
	}

	public void forEach(Consumer<AddressBookRecord> cons) {
		deserializer.forEach(rec -> {
			if (filter.test(rec)) {
				cons.accept(rec);
			}
		});
	}

	@Override
	public Collection<AddressBookRecord> all() {
		return filter(deserializer.all());
	}

	@Override
	public List<AddressBookRecord> search(Query query) {
		return filter(deserializer.search(query));
	}

	@Override
	public Collection<AddressBookRecord> byNameOrEmailPrefix(String value) {
		return filter(deserializer.byNameOrEmailPrefix(value));
	}

	@Override
	public SearchResults byKind(List<String> kinds, int offset, int limit) {
		return deserializer.byKind(kinds, offset, limit, filter);
	}

	private List<AddressBookRecord> filter(Collection<AddressBookRecord> records) {
		return records.stream().filter(filter).collect(Collectors.toList());
	}

}
