package net.bluemind.milter.impl;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.james.mime4j.dom.address.Address;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.Group;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.RawField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.netflix.spectator.api.Registry;
import com.sendmail.jilter.JilterEOMActions;
import com.sendmail.jilter.JilterHandler;
import com.sendmail.jilter.JilterStatus;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.domain.api.Domain;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.mailflow.api.ExecutionMode;
import net.bluemind.mailflow.api.MailRuleActionAssignment;
import net.bluemind.mailflow.api.MailflowRouting;
import net.bluemind.mailflow.common.api.SendingAs;
import net.bluemind.mailflow.rbe.IClientContext;
import net.bluemind.mailflow.rbe.MailflowRuleEngine;
import net.bluemind.mailflow.rbe.RuleAction;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;
import net.bluemind.milter.ClientContext;
import net.bluemind.milter.IMilterListener;
import net.bluemind.milter.IMilterListenerFactory;
import net.bluemind.milter.MilterHeaders;
import net.bluemind.milter.MilterInstanceID;
import net.bluemind.milter.SmtpAddress;
import net.bluemind.milter.Status;
import net.bluemind.milter.Status.StatusCode;
import net.bluemind.milter.action.MilterAction;
import net.bluemind.milter.action.MilterActionException;
import net.bluemind.milter.action.MilterPreAction;
import net.bluemind.milter.action.UpdatedMailMessage;
import net.bluemind.milter.cache.DomainAliasCache;
import net.bluemind.mime4j.common.Mime4JHelper;

public class MilterHandler implements JilterHandler {
	private static final Logger logger = LoggerFactory.getLogger(MilterHandler.class);
	private static final Registry registry = MetricsRegistry.get();
	private static final IdFactory idFactory = new IdFactory(MetricsRegistry.get(), MilterHandler.class);

	static {
		logger.info("JMX stats registered.");
	}

	private MessageAccumulator accumulator;
	private boolean messageModified;

	private ArrayList<IMilterListener> listeners;

	public MilterHandler(Collection<IMilterListenerFactory> mlfc) {
		listeners = new ArrayList<>(mlfc.size());
		for (IMilterListenerFactory mlf : mlfc) {
			listeners.add(mlf.create());
		}
	}

	private JilterStatus getJilterStatus(Status status) {
		switch (status.statusCode()) {
		case DISCARD:
			return JilterStatus.SMFIS_DISCARD;
		case REJECT:
			return JilterStatus.SMFIS_REJECT;
		case CUSTOM:
			return JilterStatus.makeCustomStatus(status.rcode(), status.xcode(), status.messageLines());
		case CONTINUE:
		default:
			return JilterStatus.SMFIS_CONTINUE;
		}
	}

	@Override
	public JilterStatus connect(String hostname, InetAddress hostaddr, Properties properties) {
		logger.debug("connect {} {}", hostname, hostaddr);
		accumulator = new MessageAccumulator();
		accumulator.connect(hostname, hostaddr, properties);
		return JilterStatus.SMFIS_CONTINUE;
	}

	@Override
	public JilterStatus helo(String helohost, Properties properties) {
		logger.debug("helo");
		if (accumulator != null) {
			// nagios milter.pl sends helo before connect
			accumulator.helo(properties);
		}
		return JilterStatus.SMFIS_CONTINUE;
	}

	@Override
	public JilterStatus envfrom(String[] argv, Properties properties) {
		logger.debug("envfrom");
		accumulator.envfrom(argv, properties);

		Status ret = Status.getContinue();

		for (IMilterListener listener : listeners) {
			ret = listener.onEnvFrom(properties, argv[0]);

			if (ret.statusCode() != StatusCode.CONTINUE) {
				break;
			}
		}

		return getJilterStatus(ret);
	}

	@Override
	public JilterStatus envrcpt(String[] argv, Properties properties) {
		logger.debug("envrcpt");
		accumulator.envrcpt(argv, properties);

		return forEachListener(listener -> listener.onEnvRcpt(properties, argv[0]));
	}

	private Status forEachActions(JilterEOMActions eomActions) {
		// Set message as not modified
		// Use if more than one mail was sent using same SMTP connection
		messageModified = false;

		UpdatedMailMessage modifiedMail = new UpdatedMailMessage(accumulator.getProperties(), accumulator.getMessage());

		MilterPreActionsRegistry.get().forEach(action -> applyPreAction(action, modifiedMail));
		logger.debug("Applied {} milter pre-actions", MilterPreActionsRegistry.get().size());
		modifiedMail.removeHeaders.add(MilterHeaders.SIEVE_REDIRECT);

		if (messageHasNotBeenHandledByThisInstallation()) {
			int appliedActions = applyActions(modifiedMail);
			logger.debug("Applied {} milter actions", appliedActions);
			modifiedMail.addHeader(new RawField(MilterHeaders.HANDLED, MilterInstanceID.get()),
					MilterHandler.class.getName());
			modifiedMail.addHeader(new RawField(MilterHeaders.TIMESTAMP, Long.toString(MQ.clusterTime())),
					MilterHandler.class.getName());
		}

		applyMailModifications(eomActions, modifiedMail);

		return modifiedMail.errorStatus;
	}

	private boolean messageHasNotBeenHandledByThisInstallation() {
		Field field = accumulator.getMessage().getHeader().getField(MilterHeaders.HANDLED);
		return field == null || !MilterInstanceID.get().equals(field.getBody());
	}

	private void applyMailModifications(JilterEOMActions eomActions, UpdatedMailMessage modifiedMail) {
		if (!modifiedMail.bodyChangedBy.isEmpty()) {
			logger.debug("replacing body ({})", modifiedMail.bodyChangedBy);
			Path out = null;
			try {
				out = File.createTempFile("milter", ".eml").toPath();

				try (OutputStream outStream = Files.newOutputStream(out, StandardOpenOption.TRUNCATE_EXISTING,
						StandardOpenOption.WRITE)) {
					Mime4JHelper.serializeBody(modifiedMail.getBody(), outStream);
				}

				try (FileChannel asChannel = FileChannel.open(out, StandardOpenOption.READ)) {
					long fileLength = out.toFile().length();
					MappedByteBuffer asByteBuffer = asChannel.map(MapMode.READ_ONLY, 0, fileLength);
					Files.delete(out);

					eomActions.replacebody(asByteBuffer);
				}
			} catch (IOException e) {
				logger.error(e.getMessage(), e);
			}
		}

		if (!modifiedMail.removeHeaders.isEmpty()) {
			for (String header : modifiedMail.removeHeaders) {
				logger.debug("Removing header {}", header);
				try {
					eomActions.chgheader(header, 1, null);
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
			}
		}

		if (!modifiedMail.newHeaders.isEmpty()) {
			modifiedMail.newHeaders.forEach(addedHeader -> {
				logger.debug("Add Header {}: {} - by {}", addedHeader.header().getName(),
						addedHeader.header().getBody(), addedHeader.changedBy());
				try {
					eomActions.addheader(addedHeader.header().getName(), addedHeader.header().getBody());
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
			});
		}

		modifiedMail.envelopSender.ifPresent(envelopSender -> updateEnvelopSender(eomActions, envelopSender));

		if (!modifiedMail.addRcpt.isEmpty()) {
			logger.debug("Add recipients {}", modifiedMail.addRcpt);
			modifiedMail.addRcpt.forEach(r -> {
				try {
					eomActions.addrcpt(r);
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
			});
		}

		if (!modifiedMail.removeRcpt.isEmpty()) {
			logger.debug("Remove recipients {}", modifiedMail.removeRcpt);
			modifiedMail.removeRcpt.forEach(r -> {
				try {
					eomActions.delrcpt(r);
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
			});
		}
	}

	private void updateEnvelopSender(JilterEOMActions eomActions, String envelopSender) {
		logger.debug("Update envelop sender from {} to {}", accumulator.getEnvelope().getSender(), envelopSender);
		try {
			eomActions.chgfrom(envelopSender);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	private void applyPreAction(MilterPreAction action, UpdatedMailMessage modifiedMail) {
		logger.debug("Executing pre-action {}", action.getIdentifier());
		try {
			messageModified = messageModified || action.execute(modifiedMail);
		} catch (RuntimeException e) {
			registry.counter(idFactory.name("preActionsFails")).increment();
			throw e;
		}
	}

	private int applyActions(UpdatedMailMessage modifiedMail) {
		Integer executedActions = 0;

		try {
			executedActions = getSenderDomain(accumulator.getEnvelope().getSender())
					.map(d -> applyActions(d, MailflowRouting.OUTGOING, modifiedMail)).orElse(0);
			if (executedActions == 0) {
				// rcptTo from properties is empty !!!
				executedActions = getRecipientDomain(accumulator.getEnvelope().getRecipients())
						.map(d -> applyActions(d, MailflowRouting.INCOMING, modifiedMail)).orElse(0);
			}
		} catch (Exception e) {
			logger.warn("Error while applying milter actions", e);
		}

		return executedActions;
	}

	private Integer applyActions(ItemValue<Domain> domain, MailflowRouting mailflowRouting,
			UpdatedMailMessage modifiedMail) {
		Integer executedActions = 0;
		IClientContext mailflowContext = new ClientContext(domain);

		List<MailRuleActionAssignment> storedRules = RuleAssignmentCache
				.getStoredRuleAssignments(mailflowContext, domain.uid).stream()
				.filter(rule -> rule.routing == mailflowRouting || rule.routing == MailflowRouting.ALL)
				.collect(Collectors.toList());

		storedRules.addAll(
				MilterRuleActionsRegistry.get().stream().filter(rule -> rule.routing == mailflowRouting).toList());

		List<RuleAction> matches = new MailflowRuleEngine(mailflowContext).evaluate(storedRules, toBmMessage());
		for (RuleAction ruleAction : matches) {
			try {
				ExecutionMode mode = executeAction(ruleAction, mailflowContext, modifiedMail);
				executedActions++;
				if (mode == ExecutionMode.STOP_AFTER_EXECUTION
						|| modifiedMail.errorStatus.statusCode() != StatusCode.CONTINUE) {
					logger.debug("Stopping execution of Milter actions after ruleAssignment {}",
							ruleAction.assignment.uid);
					return executedActions;
				}
			} catch (MilterActionException e) {
				logger.warn("Milter action not executed : {}", e.getMessage());
			}
		}

		return executedActions;
	}

	private Optional<ItemValue<Domain>> getRecipientDomain(List<SmtpAddress> recipients) {

		if (recipients.isEmpty()) {
			logger.warn("No recipients found");
			return Optional.empty();
		}

		ItemValue<Domain> domain = DomainAliasCache.getDomain(recipients.get(0).getDomainPart());
		if (null == domain) {
			logger.warn("Cannot find domain/alias of recipient {}", recipients.get(0));
			return Optional.empty();
		}

		if (recipients.stream().map(recipient -> DomainAliasCache.getDomain(recipient.getDomainPart()))
				.filter(Objects::nonNull).anyMatch(d -> !d.uid.equals(domain.uid))) {
			logger.warn("Recipients are not in the same BlueMind domain {}", domain.uid);
			return Optional.empty();
		}

		return Optional.of(domain);
	}

	private Optional<ItemValue<Domain>> getSenderDomain(SmtpAddress sender) {
		return Optional.ofNullable(DomainAliasCache.getDomain(sender.getDomainPart()));
	}

	private net.bluemind.mailflow.common.api.Message toBmMessage() {
		net.bluemind.mailflow.common.api.Message msg = new net.bluemind.mailflow.common.api.Message();
		msg.sendingAs = getSendingAs();
		AddressList to = accumulator.getMessage().getTo();
		if (null != to) {
			msg.to = addressListToEmail(to);
		}
		AddressList cc = accumulator.getMessage().getCc();
		if (null != cc) {
			msg.cc = addressListToEmail(cc);
		}

		msg.recipients = accumulator.getEnvelope().getRecipients().stream().map(r -> r.getEmailAddress()).toList();
		msg.subject = accumulator.getMessage().getSubject();
		return msg;
	}

	private SendingAs getSendingAs() {
		SendingAs sendingAs = new SendingAs();
		if (accumulator.getMessage().getFrom() != null && !accumulator.getMessage().getFrom().isEmpty()) {
			sendingAs.from = accumulator.getMessage().getFrom().get(0).getAddress();
		}
		if (null != accumulator.getMessage().getSender()) {
			sendingAs.sender = accumulator.getMessage().getSender().getAddress();
		} else if (null != accumulator.getEnvelope().getSender()) {
			sendingAs.sender = accumulator.getEnvelope().getSender().getEmailAddress();
		}
		if (sendingAs.from == null) {
			if (sendingAs.sender != null) {
				sendingAs.from = sendingAs.sender;
			} else {
				sendingAs.from = "";
			}
		}
		if (sendingAs.sender == null) {
			sendingAs.sender = sendingAs.from;
		}
		return sendingAs;
	}

	private List<String> addressListToEmail(AddressList addressList) {
		return addressList.stream().map(this::addressToEmail).flatMap(l -> Stream.of(l.toArray(new String[0])))
				.toList();
	}

	private List<String> addressToEmail(Address address) {
		List<String> addresses = new ArrayList<>();
		if (address instanceof Group group) {
			group.getMailboxes().forEach(a -> addresses.add(a.getAddress()));
		} else {
			Mailbox mb = (Mailbox) address;
			addresses.add(mb.getAddress());
		}
		return addresses;
	}

	private ExecutionMode executeAction(RuleAction ruleAssignment, IClientContext mailflowContext,
			UpdatedMailMessage modifiedMail) {
		Optional<MilterAction> action = MilterActionsRegistry.get(ruleAssignment.assignment.actionIdentifier);
		if (!action.isPresent()) {
			logger.warn("Unable to find registered action {}", ruleAssignment.assignment.actionIdentifier);
		} else {
			logger.debug("Executing action {}", ruleAssignment.assignment.actionIdentifier);
			try {
				action.get().execute(modifiedMail, ruleAssignment.assignment.actionConfiguration,
						ruleAssignment.rule.data, mailflowContext);
				messageModified = true;
			} catch (RuntimeException e) {
				registry.counter(idFactory.name("actionsFails")).increment();
				throw e;
			}
		}
		return ruleAssignment.assignment.mode;
	}

	private JilterStatus forEachListener(Function<IMilterListener, Status> func) {
		Status ret = Status.getContinue();
		for (IMilterListener listener : listeners) {
			Status listenerRet = func.apply(listener);
			if (listenerRet != null) {
				ret = listenerRet;
			}

			if (ret.statusCode() != StatusCode.CONTINUE) {
				break;
			}
		}

		return getJilterStatus(ret);
	}

	@Override
	public JilterStatus header(String headerf, String headerv) {
		logger.debug("header {}: {}", headerf, headerv);
		accumulator.header(headerf, headerv);

		return forEachListener(listener -> listener.onHeader(headerf, headerv));
	}

	@Override
	public JilterStatus eoh() {
		logger.debug("eoh");
		accumulator.eoh();

		return forEachListener(IMilterListener::onEoh);
	}

	@Override
	public JilterStatus body(ByteBuffer bodyp) {
		logger.debug("body");
		accumulator.body(bodyp);

		return forEachListener(listener -> listener.onBody(bodyp));
	}

	@Override
	public JilterStatus eom(JilterEOMActions eomActions, Properties properties) {
		logger.debug("eom");
		accumulator.done(properties);

		Status actionStatus = forEachActions(eomActions);

		JilterStatus ret = forEachListener(
				listener -> listener.onMessage(properties, accumulator.getEnvelope(), accumulator.getMessage()));

		accumulator.reset();
		return actionStatus.statusCode() != StatusCode.CONTINUE ? getJilterStatus(actionStatus) : ret;
	}

	@Override
	public JilterStatus abort() {
		logger.debug("abort");
		accumulator.reset();
		return JilterStatus.SMFIS_CONTINUE;
	}

	@Override
	public JilterStatus close() {
		logger.debug("close");
		accumulator.reset();
		return JilterStatus.SMFIS_CONTINUE;
	}

	@Override
	public int getSupportedProcesses() {
		int supported = PROCESS_CONNECT | PROCESS_BODY | PROCESS_ENVFROM | PROCESS_ENVRCPT | PROCESS_HEADER
				| PROCESS_HELO;
		if (logger.isDebugEnabled()) {
			logger.debug("supportedProcesses: {}", Integer.toBinaryString(supported));
		}
		return supported;
	}

	@Override
	public int getRequiredModifications() {
		logger.debug("reqMods");

		return !messageModified ? SMFIF_NONE : SMFIF_CHGBODY;
	}

	public static void init() {
		// force static init
	}

}
