/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.milter;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;

import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.hornetq.client.MQ;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.milter.impl.MilterMainServer;
import net.bluemind.systemd.notify.SystemD;

public class MilterApplication implements IApplication {

	private static final Logger logger = LoggerFactory.getLogger(MilterApplication.class);
	private static final int PORT = 2500;

	private MilterMainServer server;

	@Override
	public Object start(IApplicationContext context) throws Exception {
		if (SystemD.isAvailable()) {
			SystemD.get().notifyReady();
		}
		Thread.setDefaultUncaughtExceptionHandler(
				(thread, exception) -> logger.error("Unhandled exception in {}", thread, exception));

		logger.info("Starting MILTER...");

		launchVerticles().whenComplete((v, ex) -> {
			if (ex != null) {
				logger.warn("Cannot spawn verticles", ex);
			} else {
				logger.info("Startup complete.");
			}

		});

		server = new MilterMainServer(PORT, "0.0.0.0");
		server.start();

		logger.info("End Starting MILTER...");

		return IApplication.EXIT_OK;
	}

	private CompletableFuture<Void> launchVerticles() {
		CompletableFuture<Void> future = new CompletableFuture<>();
		MQ.init(() -> VertxPlatform.spawnVerticles((e) -> {
			if (!e.succeeded()) {
				future.completeExceptionally(e.cause());
			} else {
				future.complete(null);
			}
		}));
		return future;
	}

	@Override
	public void stop() {
		try {
			server.stop();
		} catch (IOException e) {
			logger.error("Error stopping server", e);
		}
		logger.info("Shutting down.");
	}
}
