/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.keydb;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NCUtils;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class KeyDBService {
	private static final Logger logger = LoggerFactory.getLogger(KeyDBService.class);

	public void updateConf(ItemValue<Server> server) {
		INodeClient nodeClient = NodeActivator.get(server.value.address());

		List<ItemValue<Server>> servers = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM) //
				.instance(IServer.class, "default").allComplete() //
				.stream().filter(s -> s.value.tags.contains(TagDescriptor.bm_keydb.getTag())).toList();

		if (servers.size() == 1) {
			nodeClient.writeFile("/etc/keydb/bm.conf",
					getClass().getClassLoader().getResourceAsStream("templates/mono.conf"));
		} else {
			servers.forEach(keyDbSrv -> {
				List<String> replicaOf = servers.stream().filter(curSrv -> !curSrv.uid.equals(keyDbSrv.uid)) //
						.map(s -> "replicaof " + s.value.address() + " 6379") //
						.toList();
				INodeClient nodeClientSrv = NodeActivator.get(keyDbSrv.value.address());
				ByteArrayInputStream inputStream = new ByteArrayInputStream(
						processMultiConf(replicaOf).toString().getBytes());
				nodeClientSrv.writeFile("/etc/keydb/bm.conf", inputStream);
			});
		}
		NCUtils.exec(nodeClient, 90, TimeUnit.SECONDS, "systemctl", "restart", "keydb");
		VertxPlatform.eventBus().publish("redis.storage.reset", new JsonObject());
	}

	private StringWriter processMultiConf(List<String> serversIPs) {
		Configuration cfg = new Configuration(Configuration.DEFAULT_INCOMPATIBLE_IMPROVEMENTS);
		cfg.setClassForTemplateLoading(getClass(), "/templates");
		try {
			StringWriter sw = new StringWriter();
			Map<String, Object> data = new HashMap<>();
			data.put("listOfReplicas", serversIPs);
			Template multiConfTemplate = cfg.getTemplate("multi.conf");
			multiConfTemplate.process(data, sw);
			return sw;
		} catch (TemplateException | IOException e) {
			throw new ServerFault(e);
		}
	}
}
