/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.importation.commons.hooks;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.apache.directory.api.ldap.model.cursor.CursorException;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.message.BindRequest;
import org.apache.directory.api.ldap.model.message.BindRequestImpl;
import org.apache.directory.api.ldap.model.message.BindResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Strings;
import com.netflix.spectator.api.Timer;

import net.bluemind.authentication.provider.IAuthProvider;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomainSettings;
import net.bluemind.system.importation.Activator;
import net.bluemind.system.importation.commons.Parameters;
import net.bluemind.system.importation.commons.UuidMapper;
import net.bluemind.system.importation.commons.pool.LdapConnectionContext;
import net.bluemind.system.importation.commons.pool.LdapPoolByDomain;
import net.bluemind.system.importation.metrics.MetricsHolder;
import net.bluemind.system.importation.search.PagedSearchResult.LdapSearchException;
import net.bluemind.system.importation.tools.DirectoryCredential;
import net.bluemind.user.api.User;

public abstract class ImportAuthenticationService implements IAuthProvider {
	@SuppressWarnings("serial")
	public class GetDnFailure extends Exception {
	}

	private static final Logger logger = LoggerFactory.getLogger(ImportAuthenticationService.class);

	private static final Cache<UuidMapper, String> uidToDN = Caffeine.newBuilder().recordStats().initialCapacity(1024)
			.expireAfterWrite(20, TimeUnit.MINUTES).build();
	private static final Cache<String, String> dnToPass = Caffeine.newBuilder().recordStats().initialCapacity(1024)
			.expireAfterWrite(20, TimeUnit.MINUTES).build();
	private static final Cache<String, String> uuidToDn = Caffeine.newBuilder().expireAfterWrite(1, TimeUnit.HOURS)
			.recordStats().build();

	public static class CacheRegistration implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register("import-authentification-uidtodn", uidToDN);
			cr.register("import-authentification-dntopassword", dnToPass);
			cr.register("import-authentification-uuidtodn", uuidToDn);
		}
	}

	private final MetricsHolder metrics;

	protected ImportAuthenticationService(MetricsHolder metrics) {
		this.metrics = metrics;
	}

	@Override
	public int priority() {
		return 100;
	}

	@Override
	public AuthResult check(BmContext ctx, IAuthContext authContext) {
		ItemValue<User> userItem = authContext.getUser();
		if (userExistsInDB(userItem) && (userItem.externalId == null || !userItem.externalId.startsWith(getPrefix()))) {
			// User exists in BM DB but not imported from directory
			return AuthResult.UNKNOWN;
		}

		if (Strings.isNullOrEmpty(authContext.getUserPassword())) {
			logger.error("{} authentication refused null or empty password for {}", getDirectoryKind(),
					authContext.getRealUserLogin());
			return (userExistsInDB(userItem) ? AuthResult.NO : AuthResult.UNKNOWN);
		}

		ItemValue<Domain> domain = authContext.getDomain();
		Parameters parameters;
		try {
			Map<String, String> domainSettings = ctx.su().provider().instance(IDomainSettings.class, domain.uid).get();
			parameters = getParameters(domain.value, domainSettings,
					DirectoryCredential.get(domain.uid, getDirectoryCredentialPropertyName()));
		} catch (ServerFault e) {
			logger.error("Unable to load parameters for domain {}", domain.uid, e);
			return (userExistsInDB(userItem) ? AuthResult.NO : AuthResult.UNKNOWN);
		}

		if (!parameters.enabled) {
			return (userExistsInDB(userItem) ? AuthResult.NO : AuthResult.UNKNOWN);
		}

		if (userExistsInDB(userItem)) {
			return authImportedUser(domain, parameters, authContext);
		}

		return authNotImportedUser(domain, parameters, authContext);
	}

	private boolean userExistsInDB(ItemValue<User> userItem) {
		return userItem != null;
	}

	/**
	 * Authenticate directory already imported user
	 * 
	 * @param domain
	 * @param parameters
	 * @param authContext
	 * @return
	 */
	private AuthResult authImportedUser(ItemValue<Domain> domain, Parameters parameters, IAuthContext authContext) {
		String userDn;
		try {
			userDn = getUserDnFromExtId(parameters, domain, authContext.getUser());
		} catch (GetDnFailure e) {
			return AuthResult.NO;
		}

		if (userDn == null) {
			return AuthResult.NO;
		}

		String cachedPass = dnToPass.getIfPresent(userDn);
		if (cachedPass != null && cachedPass.equals(authContext.getUserPassword())) {
			logger.debug("Allowed directory user {} from dnToPass cache system", userDn);
			return AuthResult.YES;
		}

		AuthResult authResult = checkAuth(parameters, userDn, authContext.getUserPassword());
		if (AuthResult.YES == authResult) {
			dnToPass.put(userDn, authContext.getUserPassword());
		}

		return authResult;
	}

	/**
	 * Try to authenticate non-existing BlueMind user against directory to check if
	 * it exists in directory
	 * 
	 * @param domain
	 * @param parameters
	 * @param authContext
	 * @return
	 */
	private AuthResult authNotImportedUser(ItemValue<Domain> domain, Parameters parameters, IAuthContext authContext) {
		try {
			logger.info("User {} not found in database, search login in {}", authContext.getRealUserLogin(),
					getDirectoryKind());
			String userDn = getUserDnByUserLogin(parameters, domain.value.name, authContext.getRealUserLogin());

			if (userDn == null) {
				return AuthResult.UNKNOWN;
			}

			AuthResult authResult = checkAuth(parameters, userDn, authContext.getUserPassword());
			if (AuthResult.YES == authResult) {
				dnToPass.put(userDn, authContext.getUserPassword());
			}

			return authResult;
		} catch (Exception e) {
			logger.error("Unable to search for user login {} in {}", authContext.getRealUserLogin(), getDirectoryKind(),
					e);
			return AuthResult.UNKNOWN;
		}
	}

	protected AuthResult checkAuth(Parameters parameters, String userDn, String userPassword) {
		Timer authTimer = metrics.forOperation("authCheck");

		long time = metrics.clock.monotonicTime();
		AuthResult authResult = Activator.getLdapPoolByDomain().getConnectionContext(parameters)
				.map(ldapConnnectionContext -> checkAuth(ldapConnnectionContext, userDn, userPassword))
				.orElse(AuthResult.NO);
		authTimer.record(metrics.clock.monotonicTime() - time, TimeUnit.NANOSECONDS);

		return authResult;
	}

	private AuthResult checkAuth(LdapConnectionContext ldapConnnectionContext, String userDn, String userPassword) {
		try {
			BindRequest bindRequest = new BindRequestImpl();
			bindRequest.setSimple(true);
			bindRequest.setName(userDn);
			bindRequest.setCredentials(userPassword);

			long directoryBindDuration = System.currentTimeMillis();
			BindResponse bindResponse = ldapConnnectionContext.ldapCon.bind(bindRequest);
			directoryBindDuration = System.currentTimeMillis() - directoryBindDuration;

			return bindResponseToAuthResult(ldapConnnectionContext.parameters, ldapConnnectionContext, bindResponse,
					directoryBindDuration, userDn);
		} catch (Exception e) {
			logger.error("Fail to check {} authentication: {}", getDirectoryKind(), e.getMessage());
			if (ldapConnnectionContext != null) {
				ldapConnnectionContext = ldapConnnectionContext.setError();
			}

			return AuthResult.NO;
		} finally {
			releaseConnection(Activator.getLdapPoolByDomain(), ldapConnnectionContext);
		}
	}

	private String getUserDnByUserLogin(Parameters parameters, String domainName, String userLogin) {
		if (userLogin == null) {
			return null;
		}

		Timer byLoginTimer = metrics.forOperation("dnByLogin");

		LdapPoolByDomain ldapPoolByDomain = Activator.getLdapPoolByDomain();
		LdapConnectionContext ldapConCtx = null;

		long time = metrics.clock.monotonicTime();
		try {
			ldapConCtx = ldapPoolByDomain.getAuthenticatedConnectionContext(parameters).orElse(null);
			if (ldapConCtx == null) {
				logger.error("Unable to get directory connection for domain: {}, parameters: {}", domainName,
						parameters);
				return null;
			}

			String userDn = searchUserDnFromLogin(ldapConCtx, userLogin);

			byLoginTimer.record(metrics.clock.monotonicTime() - time, TimeUnit.NANOSECONDS);

			if (userDn == null) {
				logger.error("Unable to find {}@{}", userLogin, domainName);
			}

			return userDn;
		} catch (Exception e) {
			logger.error("Fail to get DN for user: {}@{}, parameters: {}", userLogin, domainName, parameters, e);
			if (ldapConCtx != null) {
				ldapConCtx.setError();
			}

			return null;
		} finally {
			if (ldapConCtx != null) {
				releaseConnection(ldapPoolByDomain, ldapConCtx);
			}
		}
	}

	private void releaseConnection(LdapPoolByDomain ldapPoolByDomain, LdapConnectionContext ldapConCtx) {
		Timer relTimer = metrics.forOperation("release");
		long time = metrics.clock.monotonicTime();

		ldapPoolByDomain.releaseConnectionContext(ldapConCtx);

		relTimer.record(metrics.clock.monotonicTime() - time, TimeUnit.NANOSECONDS);
	}

	/**
	 * @param parameters
	 * @param domain
	 * @param authContext
	 * @return null if not a directory user, else user directory DN
	 * @throws UuidSearchException
	 */
	private String getUserDnFromExtId(Parameters parameters, ItemValue<Domain> domain, ItemValue<User> userItem)
			throws GetDnFailure {
		Optional<UuidMapper> bmUserUid = getUuidMapper(userItem.externalId);
		if (!bmUserUid.isPresent()) {
			return null;
		}

		String udn = uidToDN.getIfPresent(bmUserUid.get());

		long ldSearchTime = 0;
		if (udn == null) {
			ldSearchTime = System.currentTimeMillis();

			udn = getUserDnByUuid(parameters, bmUserUid.get().getGuid());

			if (udn != null) {
				uidToDN.put(bmUserUid.get(), udn);
			}

			ldSearchTime = System.currentTimeMillis() - ldSearchTime;
		}

		if (udn == null) {
			logger.error("Unable to find DN for extId {}, user {}@{}. Time: {}ms.", bmUserUid.get().getExtId(),
					userItem.value.login, domain.value.name, ldSearchTime);
			throw new GetDnFailure();
		}

		// don't flood logs with fast searches
		if (ldSearchTime > 10) {
			logger.info("Found: {}, searched for extId {}, u: {}@{}. Time: {}ms.", udn, bmUserUid.get().getExtId(),
					userItem.value.login, domain.value.name, ldSearchTime);
		}

		return udn;
	}

	protected String getUserDnByUuid(Parameters parameters, String uuid) throws GetDnFailure {
		String ldapUserLogin = uuidToDn.getIfPresent(uuid);
		if (ldapUserLogin != null) {
			return ldapUserLogin;
		}
		Timer byUUidTimer = metrics.forOperation("dnByUUID");

		LdapPoolByDomain ldapPoolByDomain = Activator.getLdapPoolByDomain();
		LdapConnectionContext ldapConCtx = null;

		long time = metrics.clock.monotonicTime();
		try {
			ldapConCtx = ldapPoolByDomain.getAuthenticatedConnectionContext(parameters).orElseThrow(GetDnFailure::new);

			ldapUserLogin = searchUserDnFromUuid(ldapConCtx, uuid);

			if (ldapUserLogin != null) {
				uuidToDn.put(uuid, ldapUserLogin);
			}

			return ldapUserLogin;
		} catch (Exception e) {
			if (ldapConCtx != null) {
				ldapConCtx.setError();
			}

			throw new GetDnFailure();
		} finally {
			byUUidTimer.record(metrics.clock.monotonicTime() - time, TimeUnit.NANOSECONDS);

			if (ldapConCtx != null) {
				releaseConnection(ldapPoolByDomain, ldapConCtx);
			}
		}
	}

	/**
	 * Get directory kind
	 * 
	 * @return
	 */
	protected abstract String getDirectoryKind();

	/**
	 * Get directory external ID prefix
	 * 
	 * @return
	 */
	protected abstract String getPrefix();

	/**
	 * Get domain directory parameters
	 * 
	 * @param value
	 * @param domainSettings
	 * @param credential
	 * @return
	 */
	protected abstract Parameters getParameters(Domain domain, Map<String, String> domainSettings, String credential);

	/**
	 * Get user UuidMapper from user external ID
	 * 
	 * @param externalId
	 * @return
	 */
	protected abstract Optional<UuidMapper> getUuidMapper(String externalId);

	/**
	 * Search user directory DN from user login
	 * 
	 * @param ldapConCtx
	 * @param parameters
	 * @param userLogin
	 * @return
	 */
	protected abstract String searchUserDnFromLogin(LdapConnectionContext ldapConCtx, String userLogin)
			throws LdapException, CursorException, LdapSearchException;

	/**
	 * Get user directory DN from user external UUID
	 * 
	 * @param ldapConCtx
	 * @param parameters
	 * @param uuid
	 * @return
	 */
	protected abstract String searchUserDnFromUuid(LdapConnectionContext ldapConCtx, String uuid)
			throws LdapException, CursorException, LdapSearchException;

	/**
	 * getAuthResult from directory bind response
	 * 
	 * @param parameters
	 * 
	 * @param ldapConCtx
	 * @param bindResponse
	 * @param directoryBindDuration
	 * @param userDn
	 * @return
	 */
	protected abstract AuthResult bindResponseToAuthResult(Parameters parameters, LdapConnectionContext ldapConCtx,
			BindResponse bindResponse, long directoryBindDuration, String userDn);

	/**
	 * Add directory password property to domain use credential service
	 * 
	 * @param value
	 * @return
	 */
	protected abstract String getDirectoryCredentialPropertyName();
}
