/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2018
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Suppliers;
import com.google.common.collect.Lists;

import io.netty.buffer.ByteBufUtil;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.MessageProducer;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.IReplicatedDataExpiration;
import net.bluemind.backend.mail.replica.api.MailboxRecordExpunged;
import net.bluemind.backend.mail.replica.hook.IMessageBodyHook;
import net.bluemind.backend.mail.replica.indexing.RecordIndexActivator;
import net.bluemind.backend.mail.replica.service.BodyHooks;
import net.bluemind.backend.mail.replica.service.sds.MessageBodyObjectStore;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedStore;
import net.bluemind.backend.mail.repository.IMessageBodyStore;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.sds.sync.api.SdsSyncEvent;
import net.bluemind.sds.sync.api.SdsSyncEvent.Body;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTask;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskFilter;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskStatus;
import net.bluemind.system.api.hot.upgrade.IHotUpgrade;
import net.bluemind.system.state.StateContext;

public class ReplicatedDataExpirationService implements IReplicatedDataExpiration {

	private final BmContext context;
	private final String serverUid;
	private final IMessageBodyStore bodyStore;
	private final IMailboxRecordExpungedStore expungedStore;
	private final Supplier<MessageBodyObjectStore> bodyObjectStore;
	private final IContainerStore containerStore;

	private final List<IMessageBodyHook> bodyHooks = BodyHooks.get();

	private static final Logger logger = LoggerFactory.getLogger(ReplicatedDataExpirationService.class);

	private static final MessageProducer<Body> bodydelProducer = VertxPlatform.eventBus()
			.<Body>publisher(SdsSyncEvent.BODYDEL.busName())
			.deliveryOptions(new DeliveryOptions().setLocalOnly(true).setCodecName("SdsSyncBodyCodec"));

	public ReplicatedDataExpirationService(BmContext context, String serverUid) {
		this.context = context;
		this.serverUid = serverUid;
		this.bodyStore = RepositoryProvider.instance(IMessageBodyStore.class, context);
		this.containerStore = RepositoryProvider.instance(IContainerStore.class, context, DataLocation.of(serverUid));
		this.expungedStore = RepositoryProvider.instance(IMailboxRecordExpungedStore.class, context);
		this.bodyObjectStore = Suppliers.memoize(() -> new MessageBodyObjectStore(context, DataLocation.of(serverUid)));
	}

	@Override
	public void deleteOrphanMessageBodies() {
		if (!hotupgradePassedSuccess("message-body-refcount")) {
			throw new ServerFault("NO: message-body-refcount hotupgrade is not yet passed with success");
		}
		if (!hotupgradePassedSuccess("orphan-body-record")) {
			throw new ServerFault("NO: orphan-body-record hotupgrade is not yet passed with success");
		}
		if (StateContext.getState() != SystemState.CORE_STATE_RUNNING) {
			throw new ServerFault("NO: not in a running state");
		}
		JdbcAbstractStore.doOrFail(() -> {
			List<String> deletedOrphanBodies = bodyStore.deleteOrphanBodies();

			logger.info("Deleting {} orphan message bodies", deletedOrphanBodies.size());

			if (!deletedOrphanBodies.isEmpty()) {
				RecordIndexActivator.getIndexer().ifPresent(service -> service.deleteBodyEntries(deletedOrphanBodies));
				deletedOrphanBodies.forEach(BodiesCache.bodies::invalidate);
			}

			return null;
		});
	}

	@Override
	public TaskRef deleteMessageBodiesFromObjectStore(int days) {
		if (!hotupgradePassedSuccess("message-body-refcount")) {
			throw new ServerFault("NO: message-body-refcount hotupgrade is not yet passed with success");
		}
		if (!hotupgradePassedSuccess("orphan-body-record")) {
			throw new ServerFault("NO: orphan-body-record hotupgrade is not yet passed with success");
		}
		if (StateContext.getState() != SystemState.CORE_STATE_RUNNING) {
			throw new ServerFault("NO: not in a running state");
		}

		return context.provider().instance(ITasksManager.class).run(m -> BlockingServerTask.run(m, monitor -> {
			long totalRemoved = 0;
			long removedRows = 0;
			monitor.begin(1, "Expiring expunged messages (" + days + " days) on server " + serverUid);
			MessageBodyObjectStore sdsStore = bodyObjectStore.get();
			Instant from = Instant.now().minusSeconds(TimeUnit.DAYS.toSeconds(days));
			do {
				List<String> guids = ServerFault.onException(() -> bodyStore.deletePurgedBodies(from, 10000),
						ErrorCode.SQL_ERROR);
				removedRows = guids.size();
				totalRemoved += removedRows;
				if (sdsStore != null && !guids.isEmpty()) {
					String dataLocation = sdsStore.dataLocation();
					guids.forEach(
							guid -> bodydelProducer.write(new Body(ByteBufUtil.decodeHexDump(guid), dataLocation)));
					removeFromSdsStore(monitor, sdsStore, guids);
					guids.forEach(g -> bodyHooks.stream().forEach(h -> h.preDelete(g)));
				}
			} while (removedRows > 0);
			monitor.end(true, "removed " + totalRemoved + " bodies", "");
		}));
	}

	@Override
	public void deleteExpiredExpunged(int days) {
		if (StateContext.getState() != SystemState.CORE_STATE_RUNNING) {
			throw new ServerFault("NO: not in a running state");
		}
		ServerFault.onExceptionVoid(() -> {
			logger.info("Expiring expunged messages ({} days) on server {}", days, serverUid);
			List<MailboxRecordExpunged> expiredItems;
			do {
				expiredItems = expungedStore.getExpiredItems(days);
				logger.info("Found {} message expiring to delete", expiredItems.size());

				Map<Long, List<MailboxRecordExpunged>> partitioned = expiredItems.stream()
						.collect(Collectors.groupingBy(MailboxRecordExpunged::containerId,
								Collectors.mapping(rec -> rec, Collectors.toList())));

				partitioned.entrySet().forEach(entry -> {
					List<MailboxRecordExpunged> imapUids = entry.getValue();
					Long containerId = entry.getKey();
					try {
						Container container = containerStore.get(containerId);
						logger.info("Expiring {} messages of container {}", imapUids.size(), containerId);
						context.provider().instance(IDbMailboxRecords.class, IMailReplicaUids.uniqueId(container.uid))
								.deleteImapUids(imapUids.stream().map(i -> i.imapUid).toList());
					} catch (SQLException e) {
						logger.error("Error retrieving container {}: {}", containerId, e.getMessage());
					} catch (Exception e) {
						logger.error("Error cleaning up expiring messages on container {}: {}", containerId,
								e.getMessage());
					}
					try {
						logger.info("Purge {} expunged messages of queue for container {}", imapUids.size(),
								containerId);
						for (var mre : imapUids) {
							expungedStore.deleteExpunged(mre.subtreeId.longValue(), mre.containerId.longValue(),
									mre.itemId);
						}
					} catch (Exception e) {
						logger.error("Error cleaning up expunged messages on container {}: {}", containerId,
								e.getMessage());
					}
				});
			} while (!expiredItems.isEmpty());

		}, ErrorCode.SQL_ERROR);
	}

	private void removeFromSdsStore(IServerTaskMonitor monitor, MessageBodyObjectStore sdsStore, List<String> guids) {
		logger.info("Removing {} from object storage", guids.size());
		for (List<String> partitionedGuids : Lists.partition(guids, 100)) {
			monitor.log("Removing {} objects from object storage", partitionedGuids.size());
			try {
				sdsStore.delete(partitionedGuids);
			} catch (Exception e) {
				String guidListString = partitionedGuids.stream().collect(Collectors.joining(","));
				logger.error("sdsStore.delete() failed on guids: [{}]", guidListString, e);
				monitor.log("sdsStore.delete() failed on guids: [{}]", guidListString);
			}
		}
	}

	private boolean hotupgradePassedSuccess(String operation) {
		IHotUpgrade hotupgradeApi = context.provider().instance(IHotUpgrade.class);
		List<HotUpgradeTask> tasks = hotupgradeApi.list(HotUpgradeTaskFilter.all().operation(operation));
		return (tasks.isEmpty()
				|| tasks.stream().filter(t -> t.status != HotUpgradeTaskStatus.SUCCESS).findAny().isEmpty());
	}
}
