/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service;

import java.sql.SQLException;

import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.IMailboxRecordExpunged;
import net.bluemind.backend.mail.replica.service.internal.MailboxRecordExpungedService;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedByFolderStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.repository.provider.RepositoryProvider;

public class MailboxRecordExpungedServiceFactory
		implements ServerSideServiceProvider.IServerSideServiceFactory<IMailboxRecordExpunged> {

	@Override
	public Class<IMailboxRecordExpunged> factoryClass() {
		return IMailboxRecordExpunged.class;
	}

	@Override
	public IMailboxRecordExpunged instance(BmContext context, String... params) {
		if (params == null || params.length < 1) {
			throw new ServerFault("wrong number of instance parameters");
		}

		String folderUid = params[0];

		String uid = IMailReplicaUids.mboxRecords(folderUid);
		IContainerRouteStore router = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation loc = router.routeOf(ContainerUid.of(uid));
		try {
			IContainerStore cs = RepositoryProvider.instance(IContainerStore.class, context, loc);
			Container recordsContainer = cs.get(uid);

			IMailboxes mailboxesApi = context.su().provider().instance(IMailboxes.class, recordsContainer.domainUid);
			ItemValue<Mailbox> mailbox = mailboxesApi.getComplete(recordsContainer.owner);
			if (mailbox == null) {
				throw ServerFault.notFound("mailbox of " + recordsContainer.owner + " not found");
			}
			String subtreeContainerUid = IMailReplicaUids.subtreeUid(recordsContainer.domainUid, mailbox);
			Container subtreeContainer = cs.get(subtreeContainerUid);

			IMailboxRecordExpungedByFolderStore recordStore = RepositoryProvider
					.instance(IMailboxRecordExpungedByFolderStore.class, context, recordsContainer, subtreeContainer);
			return new MailboxRecordExpungedService(context, recordStore, recordsContainer);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
