import { vi } from "vitest";
import { Request } from "node-fetch";
import fetchMock from "fetch-mock";
import { MimeType } from "@bluemind/email";
import decrypt from "../smime/decrypt";
import encrypt from "../smime/encrypt";
import verify from "../smime/verify";
import sign from "../smime/sign";
import pkcs7 from "../pkcs7";
import * as pki from "../pki/";
import { constants, enums, exceptions, helper } from "@bluemind/smime.commons";
import { readFile } from "./helpers";
import * as savePartsModule from "../smime/saveParts";

vi.mock("../smime/saveParts", () => ({
    default: vi.fn(() =>
        Promise.resolve({
            structure: { mime: "text/plain", address: "1" }
        })
    )
}));
const { EncryptError, InvalidSignatureError, SignError, UntrustedCertificateError, UnmatchedCertificateError } =
    exceptions;
const { ENCRYPTED_HEADER_NAME, SIGNED_HEADER_NAME, SMIME_ENCRYPTION_ERROR_PREFIX, SMIME_SIGNATURE_ERROR_PREFIX } =
    constants;
const { CRYPTO_HEADERS } = enums;

const { getHeaderValue, isEncrypted, isVerified } = helper;

fetchMock.mock("/session-infos", {
    login: "mathilde.michau@blue-mind.net",
    sid: "58a1ee1b-0c30-492c-a83f-4396f0a24730",
    defaultEmail: "math@devenv.blue"
});
const keyPair = {
    key: {},
    certificate: {
        validity: {
            notBefore: new Date("2022-09-25T13:43:26.000Z"),
            notAfter: new Date("2023-09-25T13:43:26.000Z")
        }
    }
};
vi.mock("../pki/", () => ({
    default: vi.fn(),
    getMyEncryptionCertificate: vi.fn(() => Promise.resolve(keyPair.certificate)),
    getMyDecryptionKeyPair: vi.fn(() => Promise.resolve(keyPair)),
    getAllDecryptionKeyPairs: vi.fn(() => Promise.resolve([])),
    getMySignatureKeyPair: vi.fn(() => Promise.resolve(keyPair)),
    getCertificate: vi.fn(),
    checkCertificate: vi.fn()
}));
vi.mock("@bluemind/mime", async () => {
    return {
        ...(await vi.importActual("@bluemind/mime")),
        MimeBuilder: vi.fn().mockImplementation(() => ({
            build: () => "dummy structure"
        }))
    };
});

const mockUploadPart = vi.fn(() => Promise.resolve("address"));

vi.mock(import("@bluemind/backend.mail.api"), async importOriginal => {
    const actual = await importOriginal();
    const MailboxItemsClient = vi.fn().mockImplementation(() => ({
        fetch: () => Promise.resolve("data"),
        uploadPart: mockUploadPart
    }));
    return {
        ...actual,
        MailboxItemsClient
    };
});

vi.mock("../pkcs7", () => {
    const decrypt = vi.fn();
    const encrypt = vi.fn();
    const sign = vi.fn();
    const verify = vi.fn();
    const parseSignedEml = vi.fn(() => ({ content: "", toDigest: "", envelope: {} }));

    return {
        decrypt,
        encrypt,
        sign,
        verify,
        parseSignedEml,
        default: { decrypt, encrypt, sign, verify, parseSignedEml }
    };
});
vi.mock("../smime/cache/SMimePartCache", () => ({
    ...vi.importActual("../smime/cache/SMimePartCache"),
    getGuid: () => Promise.resolve("99")
}));

vi.mock(import("../smime/cache/SMimePartCache"), async importOriginal => {
    const actual = await importOriginal();
    const getGuid = () => Promise.resolve("99");
    return {
        ...actual,
        getGuid
    };
});

let mockCache = {};
global.caches = {
    open: () => {
        return Promise.resolve({
            put: (request, response) => {
                mockCache[request.url] = response;
            }
        });
    }
};

class MockResponse {}
class MockFetchEvent extends Event {}
global.Request = Request;
global.Response = MockResponse;
global.FetchEvent = MockFetchEvent;

const mainEncrypted = {
    value: {
        body: {
            recipients: [
                {
                    kind: "Originator",
                    dn: "math",
                    address: "math@devenv.blue"
                }
            ],
            structure: {
                address: "1",
                mime: "application/pkcs7-mime"
            },
            headers: [],
            date: 1668534530000
        },
        imapUid: 99
    }
};

const unencrypted = {
    value: {
        body: {
            recipients: [
                {
                    kind: "Originator",
                    dn: "math",
                    address: "math@devenv.blue"
                }
            ],
            structure: {
                mime: "multipart/alternative",
                address: "TEXT",
                children: [{ mime: "text/plain" }, { mime: "text/html" }]
            },
            headers: [],
            date: 1668534530
        }
    }
};

describe("smime", () => {
    let item;
    beforeEach(() => {
        item = {
            body: {
                date: 1671032461777,
                subject: "Mail",
                headers: [],
                recipients: [
                    {
                        kind: "Primary",
                        dn: "math",
                        address: "math@devenv.blue"
                    },
                    {
                        kind: "Originator",
                        dn: "math",
                        address: "math@devenv.blue"
                    }
                ],
                messageId: "<lbntihyw.j2pop9bobhc0@devenv.blue>",
                structure: {
                    mime: "multipart/alternative",
                    children: [
                        {
                            mime: "text/plain",
                            address: "06a5ccf7-4094-4c7f-8533-eb99b072b28d",
                            encoding: "quoted-printable",
                            charset: "utf-8"
                        },
                        {
                            mime: "text/html",
                            address: "9ba724be-a4b9-4679-be0c-1c5207401d38",
                            encoding: "quoted-printable",
                            charset: "utf-8"
                        }
                    ]
                }
            },
            imapUid: 424,
            flags: ["\\Seen"]
        };

        vi.clearAllMocks();
    });

    describe("isEncrypted", () => {
        test("return true if the message main part is crypted", () => {
            expect(isEncrypted(mainEncrypted.value.body.structure), "").toBe(true);
        });
        test("return true if the message main part is crypted with correct smime-type", () => {
            const structure = {
                address: "1",
                mime: "application/pkcs7-mime",
                mimeParameters: {
                    "smime-type": "enveloped-data"
                }
            };
            expect(isEncrypted(structure)).toBe(true);
        });
        test("return false if there is not crypted subpart", () => {
            expect(isEncrypted(unencrypted)).toBe(false);
        });
        test("return false if the message has incorrected smime-type", () => {
            const structure = {
                address: "1",
                mime: "application/pkcs7-mime",
                mimeParameters: {
                    "smime-type": "incorrect"
                }
            };
            expect(isEncrypted(structure)).toBe(false);
        });
    });
    describe("decrypt", () => {
        beforeAll(() => {
            fetchMock.mock("*", new Response());
        });
        beforeEach(() => {
            mainEncrypted.value.body.headers = [];
            mockCache = {};
        });
        test("adapt message body structure when the main part is encrypted", async () => {
            const { body } = await decrypt("uid", mainEncrypted);
            expect(body.structure).toEqual(expect.objectContaining({ mime: "text/plain", address: "1" }));
        });
        test("add decrypted parts content to part cache", async () => {
            const mockEmlMultipleParts = readEml("eml/unencrypted");
            pkcs7.decrypt = () => Promise.resolve(mockEmlMultipleParts);
            const savePartsSpy = vi.spyOn(savePartsModule, "default");

            await decrypt("uid", mainEncrypted);
            expect(savePartsSpy).toHaveBeenCalled();
        });
        test("add a header if the message is crypted", async () => {
            const { body } = await decrypt("uid", mainEncrypted);
            expect(getCryptoHeaderCode(body)).toBeTruthy();
        });
        test("add a header if the message is correcty decrypted", async () => {
            const { body } = await decrypt("uid", mainEncrypted);
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.OK).toBeTruthy();
        });
        test("add a header if the message cannot be decrypted because certificate is untrusted", async () => {
            pkcs7.decrypt = () => Promise.reject(new UntrustedCertificateError());

            const { body } = await decrypt("uid", mainEncrypted);
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.UNTRUSTED_CERTIFICATE).toBeTruthy();
        });
        test("add a header if the given certificate does not match any recipient", async () => {
            pkcs7.decrypt = () => Promise.reject(new UnmatchedCertificateError());

            const { body } = await decrypt("uid", mainEncrypted);
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.UNMATCHED_CERTIFICATE).toBeTruthy();
        });
        test("should succeed using a fallback key pair when initial decryption fails", async () => {
            const firstError = new Error("primary failed");

            // First call fails, second succeeds
            const decryptMock = vi
                .fn()
                .mockRejectedValueOnce(firstError) // primary key fails
                .mockResolvedValueOnce("decrypted with fallback"); // fallback works

            pkcs7.decrypt = decryptMock;

            vi.spyOn(pki, "getAllDecryptionKeyPairs").mockResolvedValue([
                { key: "fallbackKey", certificate: "fallbackCert" }
            ]);

            const { body, content } = await decrypt("uid", mainEncrypted);

            expect(content).toBe("decrypted with fallback");
            expect(decryptMock).toHaveBeenCalledTimes(2);
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.OK).toBeTruthy();
        });

        test("should fail if all key pairs (including fallbacks) fail to decrypt", async () => {
            const originalError = new Error("decryption failed");

            // Both primary and fallbacks fail
            const decryptMock = vi.fn().mockRejectedValue(originalError);
            pkcs7.decrypt = decryptMock;

            vi.spyOn(pki, "getAllDecryptionKeyPairs").mockResolvedValue([
                { key: "fallback1", certificate: "cert1" },
                { key: "fallback2", certificate: "cert2" }
            ]);

            const { body } = await decrypt("uid", mainEncrypted);

            // Expect error header set to UNKNOWN (or fallback to whatever was thrown)
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.UNKNOWN).toBeTruthy();
            expect(decryptMock).toHaveBeenCalledTimes(3); // 1 primary + 2 fallback
        });

        test("should fallback even if multiple fallback keys are needed", async () => {
            const decryptMock = vi
                .fn()
                .mockRejectedValueOnce(new Error("primary failed"))
                .mockRejectedValueOnce(new Error("fallback1 failed"))
                .mockResolvedValueOnce("decrypted by fallback2");

            pkcs7.decrypt = decryptMock;

            vi.spyOn(pki, "getAllDecryptionKeyPairs").mockResolvedValue([
                { key: "fallback1", certificate: "cert1" },
                { key: "fallback2", certificate: "cert2" }
            ]);

            const { content, body } = await decrypt("uid", mainEncrypted);

            expect(content).toBe("decrypted by fallback2");
            expect(decryptMock).toHaveBeenCalledTimes(3); // primary + 2 fallbacks
            expect(getCryptoHeaderCode(body) & CRYPTO_HEADERS.OK).toBeTruthy();
        });
    });
    describe("encrypt", () => {
        beforeEach(() => {
            pkcs7.encrypt = () => "encrypted";
        });
        test("adapt message body structure and upload new encrypted part when the main part has to be encrypted ", async () => {
            const structure = await encrypt(item, "folderUid");
            expect(mockUploadPart).toHaveBeenCalled();
            expect(structure.body.structure.address).toBe("address");
            expect(structure.body.structure.mime).toBe(MimeType.PKCS_7);
        });
        test("raise an error if the message cannot be encrypted", async () => {
            pkcs7.encrypt = () => {
                throw new EncryptError();
            };
            try {
                await encrypt(item, "folderUid");
            } catch (error) {
                expect(error).toContain(SMIME_ENCRYPTION_ERROR_PREFIX);
            }
        });
    });
    describe("sign", () => {
        beforeEach(() => {
            pkcs7.sign = vi.fn(() => Promise.resolve("b64"));
        });
        test("adapt body structure and upload full eml", async () => {
            const signedItem = await sign(item, "folderUid");
            const multipartSigned = signedItem.body.structure;
            expect(multipartSigned.mime).toBe("message/rfc822");
            expect(multipartSigned.children.length).toBe(0);
            expect(mockUploadPart).toHaveBeenCalledTimes(1);
        });
        test("raise an error if the message cannot be signed", async () => {
            try {
                pkcs7.sign = () => {
                    throw new SignError();
                };
                await sign(item, "folderUid");
            } catch (error) {
                expect(error).toContain(SMIME_SIGNATURE_ERROR_PREFIX);
                expect(error).toContain(`${CRYPTO_HEADERS.SIGN_FAILURE}`);
            }
        });
    });
    describe("verify", () => {
        const getEml = () => readEml("eml/signed_only/valid");
        beforeEach(() => {
            vi.mocked(pkcs7.verify).mockResolvedValue(undefined);
        });
        test("add a OK header if item is successfuly verified", async () => {
            vi.spyOn(savePartsModule, "default");
            const itemValue = { value: item };
            const verified = await verify(itemValue, "folderUid", getEml);
            expect(isVerified(verified.headers)).toBe(true);
        });
        test("add a KO header if item cant be verified", async () => {
            vi.spyOn(pkcs7, "verify").mockRejectedValue(new InvalidSignatureError());

            const itemValue = { value: item };
            const verified = await verify(itemValue, "folderUid", getEml);
            expect(isVerified(verified.headers)).toBe(false);
            const headerValue = getHeaderValue(verified.headers, SIGNED_HEADER_NAME); // ← FIXED LINE
            expect(Boolean(headerValue & CRYPTO_HEADERS.INVALID_SIGNATURE)).toBe(true);
        });

        test("save parts is called even if verify fails", async () => {
            const savePartsSpy = vi.spyOn(savePartsModule, "default");
            vi.spyOn(pkcs7, "verify").mockRejectedValue(new InvalidSignatureError());

            const itemValue = { value: item };
            await verify(itemValue, "folderUid", getEml);
            expect(savePartsSpy).toHaveBeenCalled();
        });
    });
});

function readEml(file) {
    return readFile(`${file}.eml`);
}

function getCryptoHeaderCode(body) {
    const code = body.headers.find(h => h.name === ENCRYPTED_HEADER_NAME).values[0];
    return parseInt(code);
}
