import FDBFactory from "fake-indexeddb/lib/FDBFactory";
import fetchMock from "fetch-mock";
import { RevocationResult } from "@bluemind/smime.cacerts.api";

import { enums, exceptions } from "@bluemind/smime.commons";

import {
    clear,
    getMySignatureKeyPair,
    getMyDecryptionKeyPair,
    getAllDecryptionKeyPairs,
    getMyStatus,
    setMyPrivateKeys,
    setMyCertificates
} from "../../pki";
import db from "../../pki/SMimePkiDB";
import { readFile } from "../helpers";

const { PKIStatus } = enums;
const { MyKeyPairNotFoundError, InvalidCertificateError } = exceptions;

fetchMock.mock("/session-infos", { userId: "baz", domain: "foo.bar" });

const privateKey = readFile("privateKeys/adaKey.pem");
const adaNoUsageCert = readFile("certificates/adaNoUsage.pem");
const adaSignCert = readFile("certificates/adaSign.pem");
const adaEncryptCert = readFile("certificates/adaEncrypt.pem");
const adaOldNoUsageCert = readFile("certificates/ada2024.pem");

class MockedKeyAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([privateKey]));
    }
}
class MockedCertNoUsageAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([adaNoUsageCert]));
    }
}

class MockedOldAndNewNoUsageAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([adaNoUsageCert, adaOldNoUsageCert]));
    }
}
class MockedSignCertAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([adaSignCert]));
    }
}
class MockedEncryptCertAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([adaEncryptCert]));
    }
}
class TwoCertificatesAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([adaEncryptCert, adaSignCert]));
    }
}
class MockInvalidCertAsBlob extends Blob {
    text() {
        return Promise.resolve("invalid");
    }
}
class EmptyBlob extends Blob {
    text() {
        return Promise.resolve("");
    }
}

const basicCA = readFile("certificates/basicCA.crt");
const aliceCA = readFile("certificates/adaCa.crt"); // alice CA cert from RFC9216
const anyExtendedKeyUsageCert = readFile("certificates/anyExtendedKeyUsage.crt"); // self-signed

vi.mock("../../pki/SMimePkiDB");
import { resetCaCerts } from "../../pki/checkCertificate";

describe("pki", () => {
    beforeEach(() => {
        clear();
        global.indexedDB = new FDBFactory();
        fetchMock.reset();

        fetchMock.mock("/session-infos", { userId: "baz", domain: "foo.bar" });
        fetchMock.mock(
            "end:/api/smime_cacerts/smime_cacerts%3Adomain_foo.bar/_all",
            [{ value: { cert: aliceCA } }, { value: { cert: basicCA } }, { value: { cert: anyExtendedKeyUsageCert } }],
            { overwriteRoutes: true }
        );
        fetchMock.mock("end:/api/smime_revocation/foo.bar/revoked_clients", [
            {
                status: RevocationResult.RevocationStatus.NOT_REVOKED,
                revocation: { serialNumber: "myCertSerialNumber" }
            }
        ]);
    });

    afterEach(() => {
        resetCaCerts();
        fetchMock.reset();
        vi.useRealTimers();
    });

    describe("getMySignatureKeyPair", () => {
        test("returns signature key pair when certificate with signature usage is stored", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            const keyPair = await getMySignatureKeyPair();
            expect(keyPair.certificate).toBeTruthy();
            expect(keyPair.key).toBeTruthy();
        });

        test("throws MyKeyPairNotFoundError when no valid key pair found", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedEncryptCertAsBlob()); // encrypt cert only

            await expect(getMySignatureKeyPair()).rejects.toThrow(MyKeyPairNotFoundError);
        });

        test("accepts validity date parameter", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            const futureDate = new Date("2025-01-01");
            const keyPair = await getMySignatureKeyPair(futureDate);
            expect(keyPair.certificate).toBeTruthy();
            expect(keyPair.key).toBeTruthy();
        });
    });

    describe("getMyDecryptionKeyPair", () => {
        test("returns encryption key pair when certificate with encryption usage is stored", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedEncryptCertAsBlob());

            const keyPair = await getMyDecryptionKeyPair();
            expect(keyPair.certificate).toBeTruthy();
            expect(keyPair.key).toBeTruthy();
        });

        test("throws MyCertificateNotFoundError when no valid certificate found", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob()); // sign cert only

            await expect(getMyDecryptionKeyPair()).rejects.toThrow(MyKeyPairNotFoundError);
        });

        test("works with certificate that has no specified usage", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedCertNoUsageAsBlob());

            const keyPair = await getMyDecryptionKeyPair();
            expect(keyPair.certificate).toBeTruthy();
            expect(keyPair.key).toBeTruthy();
        });
    });

    describe("getAllDecryptionKeyPairs", () => {
        test("returns all valid encryption key pairs", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedOldAndNewNoUsageAsBlob());

            const keyPairs2025 = await getAllDecryptionKeyPairs(new Date("2025-01-01"));
            expect(keyPairs2025.length).toEqual(1);

            const keyPairs2024 = await getAllDecryptionKeyPairs(new Date("2024-02-01"));
            expect(keyPairs2024.length).toEqual(1);
        });

        test("returns empty array when no valid certificates found", async () => {
            db.getPrivateKeys = () => Promise.resolve(new EmptyBlob());
            db.getCertificates = () => Promise.resolve(new MockedEncryptCertAsBlob());
            const old = new Date("2023-03-17");
            const keyPairs = await getAllDecryptionKeyPairs(old);
            expect(keyPairs).toEqual([]);
        });
    });

    describe("getMyStatus", () => {
        test("returns EMPTY when no certificates or keys are stored", async () => {
            db.getPrivateKeys = () => Promise.resolve(undefined);
            db.getCertificates = () => Promise.resolve(undefined);

            const status = await getMyStatus();
            expect(status).toBe(PKIStatus.EMPTY);
        });

        test("returns OK when both signature and encryption certificates are available", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new TwoCertificatesAsBlob());

            const status = await getMyStatus();
            expect(status & PKIStatus.SIGNATURE_OK).toBe(PKIStatus.SIGNATURE_OK);
            expect(status & PKIStatus.ENCRYPTION_OK).toBe(PKIStatus.ENCRYPTION_OK);
        });

        test("returns SIGNATURE_OK when only signature certificate is available", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            const status = await getMyStatus();
            expect(status & PKIStatus.SIGNATURE_OK).toBe(PKIStatus.SIGNATURE_OK);
            expect(status & PKIStatus.ENCRYPTION_OK).not.toBe(PKIStatus.ENCRYPTION_OK);
        });

        test("returns ENCRYPTION_OK when only encryption certificate is available", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedEncryptCertAsBlob());

            const status = await getMyStatus();
            expect(status & PKIStatus.ENCRYPTION_OK).toBe(PKIStatus.ENCRYPTION_OK);
            expect(status & PKIStatus.SIGNATURE_OK).not.toBe(PKIStatus.SIGNATURE_OK);
        });

        test("handles certificate with no specified usage", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedCertNoUsageAsBlob());

            const status = await getMyStatus();
            expect(status & PKIStatus.SIGNATURE_OK).toBe(PKIStatus.SIGNATURE_OK);
            expect(status & PKIStatus.ENCRYPTION_OK).toBe(PKIStatus.ENCRYPTION_OK);
        });
    });

    describe("setMyPrivateKeys", () => {
        test("calls db.setPrivateKeys with provided blob", async () => {
            db.setPrivateKeys = vi.fn();
            const blob = new MockedKeyAsBlob();

            await setMyPrivateKeys(blob);
            expect(db.setPrivateKeys).toHaveBeenCalledWith(blob);
        });
    });

    describe("setMyCertificates", () => {
        test("calls db.setCertificates with provided blob", async () => {
            db.setCertificates = vi.fn();
            const blob = new MockedCertNoUsageAsBlob();

            await setMyCertificates(blob);
            expect(db.setCertificates).toHaveBeenCalledWith(blob);
        });
    });

    describe("clear", () => {
        test("clears database and resets cache", async () => {
            db.clearMyCertsAndKeys = vi.fn();
            db.clearRevocations = vi.fn();

            // First load some data into cache
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedCertNoUsageAsBlob());
            await getMySignatureKeyPair();

            // Clear and verify
            await clear();
            expect(db.clearMyCertsAndKeys).toHaveBeenCalled();
            expect(db.clearRevocations).toHaveBeenCalled();

            // Verify cache is cleared by checking if db methods are called again
            db.getPrivateKeys = vi.fn(() => Promise.resolve(new MockedKeyAsBlob()));
            db.getCertificates = vi.fn(() => Promise.resolve(new MockedCertNoUsageAsBlob()));
            await getMySignatureKeyPair();
            expect(db.getPrivateKeys).toHaveBeenCalled();
            expect(db.getCertificates).toHaveBeenCalled();
        });
    });

    describe("caching behavior", () => {
        test("signature key pair uses cache on subsequent calls", async () => {
            db.getPrivateKeys = vi.fn(() => Promise.resolve(new MockedKeyAsBlob()));
            db.getCertificates = vi.fn(() => Promise.resolve(new MockedSignCertAsBlob()));

            await getMySignatureKeyPair();
            await getMySignatureKeyPair();
            await getMySignatureKeyPair();

            // Should only be called once due to caching
            expect(db.getPrivateKeys).toHaveBeenCalledOnce();
            expect(db.getCertificates).toHaveBeenCalledOnce();
        });

        test("encryption key pair uses cache on subsequent calls", async () => {
            db.getPrivateKeys = vi.fn(() => Promise.resolve(new MockedKeyAsBlob()));
            db.getCertificates = vi.fn(() => Promise.resolve(new MockedEncryptCertAsBlob()));

            await getMyDecryptionKeyPair();
            await getMyDecryptionKeyPair();
            await getMyDecryptionKeyPair();

            expect(db.getPrivateKeys).toHaveBeenCalledOnce();
            expect(db.getCertificates).toHaveBeenCalledOnce();
        });

        test("different usage types have separate cache entries", async () => {
            db.getPrivateKeys = vi.fn(() => Promise.resolve(new MockedKeyAsBlob()));
            db.getCertificates = vi.fn(() => Promise.resolve(new TwoCertificatesAsBlob()));

            await getMySignatureKeyPair();
            await getMyDecryptionKeyPair();

            // Each usage type should load once
            expect(db.getPrivateKeys).toHaveBeenCalledTimes(2);
            expect(db.getCertificates).toHaveBeenCalledTimes(2);
        });
    });

    describe("error handling", () => {
        test("handles invalid certificate data gracefully", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockInvalidCertAsBlob());

            await expect(getMySignatureKeyPair()).rejects.toThrow(InvalidCertificateError);
        });

        test("handles missing private key gracefully", async () => {
            db.getPrivateKeys = () => Promise.resolve(new EmptyBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            await expect(getMySignatureKeyPair()).rejects.toThrow(MyKeyPairNotFoundError);
        });

        test("handles certificate validity date checks", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            const farFutureDate = new Date("2030-01-01");

            await expect(getMySignatureKeyPair(farFutureDate)).rejects.toThrow(MyKeyPairNotFoundError);
        });
    });
});
