/* eslint-disable @typescript-eslint/no-empty-function */
import fs from "fs";
import forge from "node-forge";
import path from "path";
import { envelope, exceptions } from "@bluemind/smime.commons";
import { base64ToArrayBuffer } from "@bluemind/arraybuffer";

import extractSignedData from "../smime/helpers/SMimeSignedDataParser";
import { readFile } from "./helpers";
import { checkSignatureValidity, checkMessageIntegrity } from "../pkcs7/verify/verify";
import pkcs7 from "../pkcs7/";
import { MessageBody } from "@bluemind/backend.mail.api";

vi.importActual("node-forge");
vi.mock("../pki/", async () => ({
    ...(await vi.importActual("../pki/")),
    checkCertificate: () => {}
}));
const { getSignedDataEnvelope } = envelope;
const { DecryptError, EncryptError, InvalidMessageIntegrityError, InvalidSignatureError, UnmatchedCertificateError } =
    exceptions;

class MockedBlob extends Blob {
    arrayBuffer() {
        return Promise.resolve(base64ToArrayBuffer(readFile("parts/encryptedPart.txt")));
    }
}

class MultipleRecipientsMockedBlob extends Blob {
    arrayBuffer() {
        return Promise.resolve(base64ToArrayBuffer(readFile("parts/encryptedMultiRecipients.txt")));
    }
}

const privatekeyTxt = readFile("privateKeys/privateKey.key");
const otherPrivateKey = readFile("privateKeys/otherPrivateKey.key");
const certificateTxt = readFile("certificates/certificate.crt");
const otherCertificateTxt = readFile("certificates/otherCertificate.crt");
const mockKey = forge.pki.privateKeyFromPem(privatekeyTxt);
const mockCertificate = forge.pki.certificateFromPem(certificateTxt);
const mockOtherCertificate = forge.pki.certificateFromPem(otherCertificateTxt);

describe("pkcs7", () => {
    describe("decrypt", () => {
        test("decrypt pkc7 part if the right private key is given", async () => {
            const res = await pkcs7.decrypt(new MockedBlob(), mockKey, mockCertificate);
            expect(res).toMatchSnapshot();
        });
        test("select the right recipient if multiple recipient are present", async () => {
            const res = await pkcs7.decrypt(new MultipleRecipientsMockedBlob(), mockKey, mockCertificate);
            expect(res).toMatchSnapshot();
        });

        test("raise an error if the given certificate does not match any recipient", async () => {
            await expect(pkcs7.decrypt(new MockedBlob(), mockKey, mockOtherCertificate)).rejects.toThrowError(
                UnmatchedCertificateError
            );
        });

        test("raise an error on decrypt failure", async () => {
            const mockKey = forge.pki.privateKeyFromPem(otherPrivateKey);
            await expect(pkcs7.decrypt(new MockedBlob(), mockKey, mockCertificate)).rejects.toThrowError(DecryptError);
        });
    });

    describe("verify", () => {
        const body = {
            date: new Date("2023-02-20").getTime(),
            recipients: [{ kind: MessageBody.RecipientKind.Originator, address: "test@devenv.blue" }],
            structure: {}
        };
        const eml = readSignedOnly("valid.eml");
        const { pkcs7Part: validPkcs7Part, toDigest: validToDigest } = extractSignedData(eml);
        const validEnvelope = getSignedDataEnvelope(validPkcs7Part);

        const { pkcs7Part: invalidPkcs7Part, toDigest: invalidToDigest } = extractSignedData(
            readSignedOnly("invalid_signature.eml")
        );
        const invalidSignatureEnvelope = getSignedDataEnvelope(invalidPkcs7Part);

        const corruptedEml = readSignedOnly("corrupted.eml");
        const { pkcs7Part: corruptedPkcs7Part, toDigest: corruptedToDigest } = extractSignedData(corruptedEml);
        const corruptedEnvelope = getSignedDataEnvelope(corruptedPkcs7Part);

        test("verify a valid eml", async () => {
            await expect(pkcs7.verify(validEnvelope, validToDigest, body)).resolves.toBeUndefined();
        });

        test("invalid eml throw an exception", async () => {
            invalidSignatureEnvelope.rawCapture.signature = "invalid";
            await expect(pkcs7.verify(invalidSignatureEnvelope, invalidToDigest, body)).rejects.toThrow();
        });
        test("corrupted eml throw an exception", async () => {
            await expect(pkcs7.verify(corruptedEnvelope, corruptedToDigest, body)).rejects.toThrowError();
        });

        test("check if signature matches authenticate attributes", async () => {
            (<forge.pki.rsa.PublicKey>invalidSignatureEnvelope.certificates[0].publicKey).verify = () => false;
            expect(() =>
                checkSignatureValidity(invalidSignatureEnvelope, invalidSignatureEnvelope.certificates[0])
            ).toThrowError(InvalidSignatureError);
        });

        test("check message integrity", async () => {
            expect(checkMessageIntegrity(validEnvelope, validToDigest, body.structure)).toBeUndefined();

            expect(() =>
                checkMessageIntegrity(corruptedEnvelope, corruptedToDigest, body.structure)
            ).toThrowErrorMatchingSnapshot(
                JSON.stringify({
                    code: 300
                })
            );

            expect(() => checkMessageIntegrity(corruptedEnvelope, corruptedToDigest, body.structure)).toThrowError(
                InvalidMessageIntegrityError
            );
        });
    });
    describe("encrypt", () => {
        test("encrypt message with my own certificate", () => {
            const result = pkcs7.encrypt("hello", [mockCertificate]);
            expect(result).toBeTruthy();
        });
        test("error in encrypt raise an error", async () => {
            forge.pkcs7.createEnvelopedData = vi.fn().mockImplementationOnce(() => {
                throw new Error();
            });

            expect(() => pkcs7.encrypt("hello", [mockCertificate])).toThrowError(EncryptError);
        });
    });
    describe("sign", () => {
        test("sign message returns a valid base64", async () => {
            const signed = await pkcs7.sign("blabla", mockKey, mockCertificate);

            expect(() => atob(signed)).not.toThrow();
        });
    });
});

function readSignedOnly(filename: string) {
    return fs.readFileSync(path.join(__dirname, `./data/eml/signed_only/${filename}`), "utf8");
}
