/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.directory.cql.common;

import java.sql.SQLException;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.internal.IInternalContainersFlatHierarchyMgmt;
import net.bluemind.core.container.api.internal.IInternalOwnerSubscriptionsMgmt;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.repository.IItemValueStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.repository.DirEntryNG;
import net.bluemind.directory.repository.IDirEntryStore;
import net.bluemind.directory.repository.IDirEntryStore.ItemIdAndRole;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.internal.IInCoreMailboxes;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.role.repository.IRoleStore;

public class CqlContStoreService extends ContainerStoreService<DirEntryNG> {

	private static final Logger logger = LoggerFactory.getLogger(CqlContStoreService.class);
	private final IInCoreMailboxes mboxApi;
	private final IDirEntryStore deStore;
	private final IRoleStore roleStore;
	private final BmContext ctx;

	public CqlContStoreService(BmContext context, Container container, ItemValue<Domain> domainValue) {
		super(context, container, RepositoryProvider.instance(IDirEntryStore.class, context, container));
		this.ctx = context;
		this.mboxApi = context.su().provider().instance(IInCoreMailboxes.class, domainValue.uid);
		this.deStore = (IDirEntryStore) itemValueStore;
		this.roleStore = RepositoryProvider.instance(IRoleStore.class, context, container);
	}

	public Item byMailboxName(String mbox) {
		return deStore.byMailboxName(mbox);
	}

	public Item byEmail(String email) {
		return deStore.byEmail(email);
	}

	public List<ItemIdAndRole> byKind(Kind k) {
		return deStore.byKind(k);
	}

	@Override
	protected void createValue(Item item, DirEntryNG value, IItemValueStore<DirEntryNG> itemValueStore)
			throws SQLException {
		super.createValue(item, value, itemValueStore);
		if (value.kind.hasMailbox()) {
			mboxApi.created(item.uid, value.asMailbox(), allocatedIds -> logger.info("IDs allocated {}", allocatedIds));

			IInternalOwnerSubscriptionsMgmt subMgmt = ctx.provider().instance(IInternalOwnerSubscriptionsMgmt.class,
					container.domainUid, item.uid);
			subMgmt.init();

			IInternalContainersFlatHierarchyMgmt chMgmt = ctx.provider()
					.instance(IInternalContainersFlatHierarchyMgmt.class, container.domainUid, item.uid);
			chMgmt.init();

		}

	}

	@Override
	protected void updateValue(Item item, DirEntryNG value) throws SQLException {
		DirEntryNG oldValue = itemValueStore.get(item);
		super.updateValue(item, value);
		if (value.kind.hasMailbox()) {
			mboxApi.updated(item.uid, oldValue.asMailbox(), value.asMailbox(),
					allocatedIds -> logger.info("IDs allocated {}", allocatedIds));
		}
	}

	@Override
	protected void deleteValue(Item item) throws SQLException {
		DirEntryNG oldValue = itemValueStore.get(item);
		super.deleteValue(item);
		if (oldValue.kind.hasMailbox()) {
			mboxApi.deleted(item.uid, oldValue.asMailbox());
		}
	}

	public void setRoles(Item it, Set<String> roles) {
		try {
			roleStore.set(it, roles);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public Set<String> getRoles(Item it) {
		try {
			return roleStore.get(it);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public void addMemberOf(String entryUid, String groupUid) {
		ServerFault.onExceptionVoid(() -> {
			Item deItem = getItemStore().get(entryUid);
			if (deItem != null) {
				deStore.addMemberOf(deItem, groupUid);
			}
		}, ErrorCode.SQL_ERROR);
	}

	public void rmMemberOf(String entryUid, String groupUid) {
		ServerFault.onExceptionVoid(() -> {
			Item deItem = getItemStore().get(entryUid);
			if (deItem != null) {
				deStore.rmMemberOf(deItem, groupUid);
			}
		}, ErrorCode.SQL_ERROR);
	}

}
