/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.DeleteRecordsResult;
import org.apache.kafka.clients.admin.RecordsToDelete;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerGroupMetadata;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Metric;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.errors.LockException;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.errors.TaskIdFormatException;
import org.apache.kafka.streams.errors.TaskMigratedException;
import org.apache.kafka.streams.internals.StreamsConfigUtils;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.assignment.ProcessId;
import org.apache.kafka.streams.processor.internals.AbstractTask;
import org.apache.kafka.streams.processor.internals.ActiveTaskCreator;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.StandbyTask;
import org.apache.kafka.streams.processor.internals.StandbyTaskCreator;
import org.apache.kafka.streams.processor.internals.StateDirectory;
import org.apache.kafka.streams.processor.internals.StateManagerUtil;
import org.apache.kafka.streams.processor.internals.StateUpdater;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.StreamsProducer;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskExecutor;
import org.apache.kafka.streams.processor.internals.TasksRegistry;
import org.apache.kafka.streams.processor.internals.TopologyMetadata;
import org.apache.kafka.streams.processor.internals.tasks.DefaultTaskManager;
import org.apache.kafka.streams.state.internals.OffsetCheckpoint;
import org.slf4j.Logger;

public class TaskManager {
    private static final String BUG_ERROR_MESSAGE = "This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).";
    private static final String INTERRUPTED_ERROR_MESSAGE = "Thread got interrupted. This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).";
    private final Logger log;
    private final Time time;
    private final TasksRegistry tasks;
    private final ProcessId processId;
    private final String logPrefix;
    private final Admin adminClient;
    private final StateDirectory stateDirectory;
    private final StreamsConfigUtils.ProcessingMode processingMode;
    private final ChangelogReader changelogReader;
    private final TopologyMetadata topologyMetadata;
    private final TaskExecutor taskExecutor;
    private Consumer<byte[], byte[]> mainConsumer;
    private DeleteRecordsResult deleteRecordsResult;
    private boolean rebalanceInProgress = false;
    private final Set<TaskId> lockedTaskDirectories = new HashSet<TaskId>();
    private final ActiveTaskCreator activeTaskCreator;
    private final StandbyTaskCreator standbyTaskCreator;
    private final StateUpdater stateUpdater;
    private final DefaultTaskManager schedulingTaskManager;

    TaskManager(Time time, ChangelogReader changelogReader, ProcessId processId, String logPrefix, ActiveTaskCreator activeTaskCreator, StandbyTaskCreator standbyTaskCreator, TasksRegistry tasks, TopologyMetadata topologyMetadata, Admin adminClient, StateDirectory stateDirectory, StateUpdater stateUpdater, DefaultTaskManager schedulingTaskManager) {
        this.time = time;
        this.processId = processId;
        this.logPrefix = logPrefix;
        this.adminClient = adminClient;
        this.stateDirectory = stateDirectory;
        this.changelogReader = changelogReader;
        this.topologyMetadata = topologyMetadata;
        this.activeTaskCreator = activeTaskCreator;
        this.standbyTaskCreator = standbyTaskCreator;
        this.processingMode = topologyMetadata.processingMode();
        LogContext logContext = new LogContext(logPrefix);
        this.log = logContext.logger(this.getClass());
        this.stateUpdater = stateUpdater;
        this.schedulingTaskManager = schedulingTaskManager;
        this.tasks = tasks;
        this.taskExecutor = new TaskExecutor(this.tasks, this, topologyMetadata.taskExecutionMetadata(), logContext);
    }

    void setMainConsumer(Consumer<byte[], byte[]> mainConsumer) {
        this.mainConsumer = mainConsumer;
    }

    public double totalProducerBlockedTime() {
        return this.activeTaskCreator.totalProducerBlockedTime();
    }

    public ProcessId processId() {
        return this.processId;
    }

    public TopologyMetadata topologyMetadata() {
        return this.topologyMetadata;
    }

    ConsumerGroupMetadata consumerGroupMetadata() {
        return this.mainConsumer.groupMetadata();
    }

    void consumerCommitSync(Map<TopicPartition, OffsetAndMetadata> offsets) {
        this.mainConsumer.commitSync(offsets);
    }

    StreamsProducer streamsProducerForTask(TaskId taskId) {
        return this.activeTaskCreator.streamsProducerForTask(taskId);
    }

    StreamsProducer threadProducer() {
        return this.activeTaskCreator.threadProducer();
    }

    boolean rebalanceInProgress() {
        return this.rebalanceInProgress;
    }

    void handleRebalanceStart(Set<String> subscribedTopics) {
        this.topologyMetadata.addSubscribedTopicsFromMetadata(subscribedTopics, this.logPrefix);
        this.tryToLockAllNonEmptyTaskDirectories();
        this.rebalanceInProgress = true;
    }

    void handleRebalanceComplete() {
        if (this.stateUpdater == null) {
            this.mainConsumer.pause(this.mainConsumer.assignment());
        } else {
            Set partitionsNotToPause = this.tasks.allNonFailedTasks().stream().flatMap(task -> task.inputPartitions().stream()).collect(Collectors.toSet());
            HashSet<TopicPartition> partitionsToPause = new HashSet<TopicPartition>(this.mainConsumer.assignment());
            partitionsToPause.removeAll(partitionsNotToPause);
            this.mainConsumer.pause(partitionsToPause);
        }
        this.releaseLockedUnassignedTaskDirectories();
        this.rebalanceInProgress = false;
    }

    boolean handleCorruption(Set<TaskId> corruptedTasks) {
        HashSet<TaskId> activeTasks = new HashSet<TaskId>(this.tasks.activeTaskIds());
        this.maybeLockTasks(activeTasks);
        HashSet<Task> corruptedActiveTasks = new HashSet<Task>();
        HashSet<Task> corruptedStandbyTasks = new HashSet<Task>();
        for (TaskId taskId : corruptedTasks) {
            Task task = this.tasks.task(taskId);
            if (task.isActive()) {
                corruptedActiveTasks.add(task);
                continue;
            }
            corruptedStandbyTasks.add(task);
        }
        this.closeDirtyAndRevive(corruptedStandbyTasks, true);
        try {
            Collection tasksToCommit = this.tasks.allTasksPerId().values().stream().filter(t -> t.state() == Task.State.RUNNING).filter(t -> !corruptedTasks.contains(t.id())).collect(Collectors.toSet());
            this.commitTasksAndMaybeUpdateCommittableOffsets(tasksToCommit, new HashMap<Task, Map<TopicPartition, OffsetAndMetadata>>());
        }
        catch (TaskCorruptedException e) {
            this.log.info("Some additional tasks were found corrupted while trying to commit, these will be added to the tasks to clean and revive: {}", e.corruptedTasks());
            corruptedActiveTasks.addAll(this.tasks.tasks(e.corruptedTasks()));
        }
        catch (TimeoutException e) {
            this.log.info("Hit TimeoutException when committing all non-corrupted tasks, these will be closed and revived");
            HashSet<Task> uncorruptedTasks = new HashSet<Task>(this.tasks.activeTasks());
            uncorruptedTasks.removeAll(corruptedActiveTasks);
            this.closeDirtyAndRevive(uncorruptedTasks, false);
        }
        this.closeDirtyAndRevive(corruptedActiveTasks, true);
        this.maybeUnlockTasks(activeTasks);
        return !corruptedActiveTasks.isEmpty();
    }

    private void closeDirtyAndRevive(Collection<Task> taskWithChangelogs, boolean markAsCorrupted) {
        for (Task task : taskWithChangelogs) {
            if (task.state() != Task.State.CLOSED) {
                Set<TopicPartition> corruptedPartitions = task.changelogPartitions();
                if (markAsCorrupted && this.stateUpdater == null) {
                    task.markChangelogAsCorrupted(corruptedPartitions);
                }
                try {
                    task.prepareCommit();
                }
                catch (RuntimeException swallow) {
                    this.log.error("Error flushing cache for corrupted task {} ", (Object)task.id(), (Object)swallow);
                }
                try {
                    task.suspend();
                    if (markAsCorrupted) {
                        task.postCommit(true);
                    }
                }
                catch (RuntimeException swallow) {
                    this.log.error("Error suspending corrupted task {} ", (Object)task.id(), (Object)swallow);
                }
                task.closeDirty();
            }
            if (task.isActive()) {
                Set<TopicPartition> taskInputPartitions;
                Set<TopicPartition> currentAssignment = this.mainConsumer.assignment();
                Set<TopicPartition> assignedToPauseAndReset = Utils.intersection(HashSet::new, currentAssignment, taskInputPartitions = task.inputPartitions());
                if (!assignedToPauseAndReset.equals(taskInputPartitions)) {
                    this.log.warn("Expected the current consumer assignment {} to contain the input partitions {}. Will proceed to recover.", currentAssignment, taskInputPartitions);
                }
                task.addPartitionsForOffsetReset(assignedToPauseAndReset);
            }
            if (this.stateUpdater != null) {
                this.tasks.removeTask(task);
            }
            task.revive();
            if (this.stateUpdater == null) continue;
            this.tasks.addPendingTasksToInit(Collections.singleton(task));
        }
    }

    public void handleAssignment(Map<TaskId, Set<TopicPartition>> activeTasks, Map<TaskId, Set<TopicPartition>> standbyTasks) {
        this.log.info("Handle new assignment with:\n\tNew active tasks: {}\n\tNew standby tasks: {}\n\tExisting active tasks: {}\n\tExisting standby tasks: {}", new Object[]{activeTasks.keySet(), standbyTasks.keySet(), this.activeTaskIds(), this.standbyTaskIds()});
        this.topologyMetadata.addSubscribedTopicsFromAssignment(activeTasks.values().stream().flatMap(Collection::stream).collect(Collectors.toSet()), this.logPrefix);
        HashMap<TaskId, Set<TopicPartition>> activeTasksToCreate = new HashMap<TaskId, Set<TopicPartition>>(activeTasks);
        HashMap<TaskId, Set<TopicPartition>> standbyTasksToCreate = new HashMap<TaskId, Set<TopicPartition>>(standbyTasks);
        HashMap<Task, Set<TopicPartition>> tasksToRecycle = new HashMap<Task, Set<TopicPartition>>();
        TreeSet<Task> tasksToCloseClean = new TreeSet<Task>(Comparator.comparing(Task::id));
        Set<TaskId> tasksToLock = this.tasks.allTaskIds().stream().filter(x -> activeTasksToCreate.containsKey(x) || standbyTasksToCreate.containsKey(x)).collect(Collectors.toSet());
        this.maybeLockTasks(tasksToLock);
        this.tasks.clearPendingTasksToCreate();
        this.tasks.addPendingActiveTasksToCreate(this.pendingTasksToCreate(activeTasksToCreate));
        this.tasks.addPendingStandbyTasksToCreate(this.pendingTasksToCreate(standbyTasksToCreate));
        LinkedHashMap<TaskId, RuntimeException> failedTasks = new LinkedHashMap<TaskId, RuntimeException>();
        if (this.stateUpdater == null) {
            this.handleTasksWithoutStateUpdater(activeTasksToCreate, standbyTasksToCreate, tasksToRecycle, tasksToCloseClean);
        } else {
            this.handleTasksWithStateUpdater(activeTasksToCreate, standbyTasksToCreate, tasksToRecycle, tasksToCloseClean, failedTasks);
            failedTasks.putAll(this.collectExceptionsAndFailedTasksFromStateUpdater());
        }
        Map<TaskId, RuntimeException> taskCloseExceptions = this.closeAndRecycleTasks(tasksToRecycle, tasksToCloseClean);
        this.maybeUnlockTasks(tasksToLock);
        failedTasks.putAll(taskCloseExceptions);
        this.maybeThrowTaskExceptions(failedTasks);
        this.createNewTasks(activeTasksToCreate, standbyTasksToCreate);
    }

    private void maybeThrowTaskExceptions(Map<TaskId, RuntimeException> taskExceptions) {
        if (!taskExceptions.isEmpty()) {
            this.log.error("Get exceptions for the following tasks: {}", taskExceptions);
            HashSet<TaskId> aggregatedCorruptedTaskIds = new HashSet<TaskId>();
            StreamsException lastFatal = null;
            TaskMigratedException lastTaskMigrated = null;
            for (Map.Entry<TaskId, RuntimeException> entry : taskExceptions.entrySet()) {
                TaskId taskId = entry.getKey();
                RuntimeException exception = entry.getValue();
                if (exception instanceof StreamsException) {
                    if (exception instanceof TaskMigratedException) {
                        lastTaskMigrated = (TaskMigratedException)exception;
                        continue;
                    }
                    if (exception instanceof TaskCorruptedException) {
                        this.log.warn("Encounter corrupted task " + taskId + ", will group it with other corrupted tasks and handle together", (Throwable)exception);
                        aggregatedCorruptedTaskIds.add(taskId);
                        continue;
                    }
                    ((StreamsException)exception).setTaskId(taskId);
                    lastFatal = (StreamsException)exception;
                    continue;
                }
                if (exception instanceof KafkaException) {
                    lastFatal = new StreamsException(exception, taskId);
                    continue;
                }
                lastFatal = new StreamsException("Encounter unexpected fatal error for task " + taskId, exception, taskId);
            }
            if (lastFatal != null) {
                throw lastFatal;
            }
            if (lastTaskMigrated != null) {
                throw lastTaskMigrated;
            }
            throw new TaskCorruptedException(aggregatedCorruptedTaskIds);
        }
    }

    private void createNewTasks(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate) {
        Collection<Task> newActiveTasks = this.activeTaskCreator.createTasks(this.mainConsumer, activeTasksToCreate);
        Collection<Task> newStandbyTasks = this.standbyTaskCreator.createTasks(standbyTasksToCreate);
        if (this.stateUpdater == null) {
            this.tasks.addActiveTasks(newActiveTasks);
            this.tasks.addStandbyTasks(newStandbyTasks);
        } else {
            this.tasks.addPendingTasksToInit(newActiveTasks);
            this.tasks.addPendingTasksToInit(newStandbyTasks);
        }
    }

    private void handleTasksWithoutStateUpdater(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate, Map<Task, Set<TopicPartition>> tasksToRecycle, Set<Task> tasksToCloseClean) {
        for (Task task : this.tasks.allTasks()) {
            TaskId taskId = task.id();
            if (activeTasksToCreate.containsKey(taskId)) {
                if (task.isActive()) {
                    Set<TopicPartition> topicPartitions = activeTasksToCreate.get(taskId);
                    if (this.tasks.updateActiveTaskInputPartitions(task, topicPartitions)) {
                        task.updateInputPartitions(topicPartitions, this.topologyMetadata.nodeToSourceTopics(task.id()));
                    }
                    task.resume();
                } else {
                    tasksToRecycle.put(task, activeTasksToCreate.get(taskId));
                }
                activeTasksToCreate.remove(taskId);
                continue;
            }
            if (standbyTasksToCreate.containsKey(taskId)) {
                if (!task.isActive()) {
                    this.updateInputPartitionsOfStandbyTaskIfTheyChanged(task, standbyTasksToCreate.get(taskId));
                    task.resume();
                } else {
                    tasksToRecycle.put(task, standbyTasksToCreate.get(taskId));
                }
                standbyTasksToCreate.remove(taskId);
                continue;
            }
            tasksToCloseClean.add(task);
        }
    }

    private void updateInputPartitionsOfStandbyTaskIfTheyChanged(Task task, Set<TopicPartition> inputPartitions) {
        if (!task.inputPartitions().equals(inputPartitions)) {
            task.updateInputPartitions(inputPartitions, this.topologyMetadata.nodeToSourceTopics(task.id()));
        }
    }

    private void handleTasksWithStateUpdater(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate, Map<Task, Set<TopicPartition>> tasksToRecycle, Set<Task> tasksToCloseClean, Map<TaskId, RuntimeException> failedTasks) {
        this.handleTasksPendingInitialization();
        this.handleRestoringAndUpdatingTasks(activeTasksToCreate, standbyTasksToCreate, failedTasks);
        this.handleRunningAndSuspendedTasks(activeTasksToCreate, standbyTasksToCreate, tasksToRecycle, tasksToCloseClean);
    }

    private void handleTasksPendingInitialization() {
        for (Task task : this.tasks.drainPendingTasksToInit()) {
            task.suspend();
            task.closeClean();
        }
    }

    private void handleRunningAndSuspendedTasks(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate, Map<Task, Set<TopicPartition>> tasksToRecycle, Set<Task> tasksToCloseClean) {
        for (Task task : this.tasks.allNonFailedTasks()) {
            if (!task.isActive()) {
                throw new IllegalStateException("Standby tasks should only be managed by the state updater, but standby task " + task.id() + " is managed by the stream thread");
            }
            TaskId taskId = task.id();
            if (activeTasksToCreate.containsKey(taskId)) {
                this.handleReassignedActiveTask(task, activeTasksToCreate.get(taskId));
                activeTasksToCreate.remove(taskId);
                continue;
            }
            if (standbyTasksToCreate.containsKey(taskId)) {
                tasksToRecycle.put(task, standbyTasksToCreate.get(taskId));
                standbyTasksToCreate.remove(taskId);
                continue;
            }
            tasksToCloseClean.add(task);
        }
    }

    private void handleReassignedActiveTask(Task task, Set<TopicPartition> inputPartitions) {
        if (this.tasks.updateActiveTaskInputPartitions(task, inputPartitions)) {
            task.updateInputPartitions(inputPartitions, this.topologyMetadata.nodeToSourceTopics(task.id()));
        }
        if (task.state() == Task.State.SUSPENDED) {
            this.tasks.removeTask(task);
            task.resume();
            this.stateUpdater.add(task);
        }
    }

    private void handleRestoringAndUpdatingTasks(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate, Map<TaskId, RuntimeException> failedTasks) {
        HashMap<Task, Set<TopicPartition>> tasksToRecycleFromStateUpdater = new HashMap<Task, Set<TopicPartition>>();
        HashSet<Task> tasksToCloseCleanFromStateUpdater = new HashSet<Task>();
        HashSet<Task> tasksToCloseDirtyFromStateUpdater = new HashSet<Task>();
        this.handleTasksInStateUpdater(activeTasksToCreate, standbyTasksToCreate, tasksToRecycleFromStateUpdater, tasksToCloseCleanFromStateUpdater, tasksToCloseDirtyFromStateUpdater, failedTasks);
        tasksToRecycleFromStateUpdater.forEach((task, inputPartitions) -> this.recycleTaskFromStateUpdater((Task)task, (Set<TopicPartition>)inputPartitions, (Set<Task>)tasksToCloseDirtyFromStateUpdater, failedTasks));
        tasksToCloseCleanFromStateUpdater.forEach(task -> this.closeTaskClean((Task)task, (Set<Task>)tasksToCloseDirtyFromStateUpdater, failedTasks));
        tasksToCloseDirtyFromStateUpdater.forEach(task -> this.closeTaskDirty((Task)task, false));
    }

    private void handleTasksInStateUpdater(Map<TaskId, Set<TopicPartition>> activeTasksToCreate, Map<TaskId, Set<TopicPartition>> standbyTasksToCreate, Map<Task, Set<TopicPartition>> tasksToRecycle, Set<Task> tasksToCloseCleanFromStateUpdater, Set<Task> tasksToCloseDirtyFromStateUpdater, Map<TaskId, RuntimeException> failedTasks) {
        HashMap<TaskId, Set<TopicPartition>> newInputPartitions = new HashMap<TaskId, Set<TopicPartition>>();
        HashMap<TaskId, Set<TopicPartition>> standbyInputPartitions = new HashMap<TaskId, Set<TopicPartition>>();
        HashMap<TaskId, Set<TopicPartition>> activeInputPartitions = new HashMap<TaskId, Set<TopicPartition>>();
        LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futuresForUpdatingInputPartitions = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
        LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futuresForActiveTasksToRecycle = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
        LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futuresForStandbyTasksToRecycle = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
        LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futuresForTasksToClose = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
        for (Task task : this.stateUpdater.getTasks()) {
            CompletableFuture<StateUpdater.RemovedTaskResult> future;
            TaskId taskId = task.id();
            if (activeTasksToCreate.containsKey(taskId)) {
                if (task.isActive()) {
                    if (!task.inputPartitions().equals(activeTasksToCreate.get(taskId))) {
                        future = this.stateUpdater.remove(taskId);
                        futuresForUpdatingInputPartitions.put(taskId, future);
                        newInputPartitions.put(taskId, activeTasksToCreate.get(taskId));
                    }
                } else {
                    future = this.stateUpdater.remove(taskId);
                    futuresForStandbyTasksToRecycle.put(taskId, future);
                    activeInputPartitions.put(taskId, activeTasksToCreate.get(taskId));
                }
                activeTasksToCreate.remove(taskId);
                continue;
            }
            if (standbyTasksToCreate.containsKey(taskId)) {
                if (task.isActive()) {
                    future = this.stateUpdater.remove(taskId);
                    futuresForActiveTasksToRecycle.put(taskId, future);
                    standbyInputPartitions.put(taskId, standbyTasksToCreate.get(taskId));
                }
                standbyTasksToCreate.remove(taskId);
                continue;
            }
            future = this.stateUpdater.remove(taskId);
            futuresForTasksToClose.put(taskId, future);
        }
        this.updateInputPartitions(futuresForUpdatingInputPartitions, newInputPartitions, failedTasks);
        this.addToActiveTasksToRecycle(futuresForActiveTasksToRecycle, standbyInputPartitions, tasksToRecycle, failedTasks);
        this.addToStandbyTasksToRecycle(futuresForStandbyTasksToRecycle, activeInputPartitions, tasksToRecycle, failedTasks);
        this.addToTasksToClose(futuresForTasksToClose, tasksToCloseCleanFromStateUpdater, tasksToCloseDirtyFromStateUpdater);
    }

    private void updateInputPartitions(Map<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures, Map<TaskId, Set<TopicPartition>> newInputPartitions, Map<TaskId, RuntimeException> failedTasks) {
        this.getNonFailedTasks(futures, failedTasks).forEach(task -> {
            task.updateInputPartitions((Set)newInputPartitions.get(task.id()), this.topologyMetadata.nodeToSourceTopics(task.id()));
            this.stateUpdater.add((Task)task);
        });
    }

    private void addToActiveTasksToRecycle(Map<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures, Map<TaskId, Set<TopicPartition>> standbyInputPartitions, Map<Task, Set<TopicPartition>> tasksToRecycle, Map<TaskId, RuntimeException> failedTasks) {
        this.getNonFailedTasks(futures, failedTasks).forEach(task -> {
            Set cfr_ignored_0 = (Set)tasksToRecycle.put((Task)task, (Set<TopicPartition>)standbyInputPartitions.get(task.id()));
        });
    }

    private void addToStandbyTasksToRecycle(Map<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures, Map<TaskId, Set<TopicPartition>> activeInputPartitions, Map<Task, Set<TopicPartition>> tasksToRecycle, Map<TaskId, RuntimeException> failedTasks) {
        this.getNonFailedTasks(futures, failedTasks).forEach(task -> {
            Set cfr_ignored_0 = (Set)tasksToRecycle.put((Task)task, (Set<TopicPartition>)activeInputPartitions.get(task.id()));
        });
    }

    private Stream<Task> getNonFailedTasks(Map<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures, Map<TaskId, RuntimeException> failedTasks) {
        return futures.entrySet().stream().map(entry -> this.waitForFuture((TaskId)entry.getKey(), (CompletableFuture)entry.getValue())).filter(Objects::nonNull).map(removedTaskResult -> this.checkIfTaskFailed((StateUpdater.RemovedTaskResult)removedTaskResult, failedTasks)).filter(Objects::nonNull);
    }

    private void addToTasksToClose(Map<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures, Set<Task> tasksToCloseCleanFromStateUpdater, Set<Task> tasksToCloseDirtyFromStateUpdater) {
        futures.entrySet().stream().map(entry -> this.waitForFuture((TaskId)entry.getKey(), (CompletableFuture)entry.getValue())).filter(Objects::nonNull).forEach(removedTaskResult -> {
            if (removedTaskResult.exception().isPresent()) {
                tasksToCloseDirtyFromStateUpdater.add(removedTaskResult.task());
            } else {
                tasksToCloseCleanFromStateUpdater.add(removedTaskResult.task());
            }
        });
    }

    private Task checkIfTaskFailed(StateUpdater.RemovedTaskResult removedTaskResult, Map<TaskId, RuntimeException> failedTasks) {
        Task task = removedTaskResult.task();
        if (removedTaskResult.exception().isPresent()) {
            failedTasks.put(task.id(), removedTaskResult.exception().get());
            this.tasks.addFailedTask(task);
            return null;
        }
        return task;
    }

    private StateUpdater.RemovedTaskResult waitForFuture(TaskId taskId, CompletableFuture<StateUpdater.RemovedTaskResult> future) {
        try {
            StateUpdater.RemovedTaskResult removedTaskResult = future.get();
            if (removedTaskResult == null) {
                throw new IllegalStateException("Task " + taskId + " was not found in the state updater. " + BUG_ERROR_MESSAGE);
            }
            return removedTaskResult;
        }
        catch (ExecutionException executionException) {
            this.log.warn("An exception happened when removing task {} from the state updater. The task was added to the failed task in the state updater: ", (Object)taskId, (Object)executionException);
            return null;
        }
        catch (InterruptedException shouldNotHappen) {
            Thread.currentThread().interrupt();
            this.log.error(INTERRUPTED_ERROR_MESSAGE, (Throwable)shouldNotHappen);
            throw new IllegalStateException(INTERRUPTED_ERROR_MESSAGE, shouldNotHappen);
        }
    }

    private Map<TaskId, Set<TopicPartition>> pendingTasksToCreate(Map<TaskId, Set<TopicPartition>> tasksToCreate) {
        HashMap<TaskId, Set<TopicPartition>> pendingTasks = new HashMap<TaskId, Set<TopicPartition>>();
        Iterator<Map.Entry<TaskId, Set<TopicPartition>>> iter = tasksToCreate.entrySet().iterator();
        while (iter.hasNext()) {
            boolean taskIsOwned;
            Map.Entry<TaskId, Set<TopicPartition>> entry = iter.next();
            TaskId taskId = entry.getKey();
            boolean bl = taskIsOwned = this.tasks.allTaskIds().contains(taskId) || this.stateUpdater != null && this.stateUpdater.getTasks().stream().anyMatch(task -> task.id() == taskId);
            if (taskId.topologyName() == null || taskIsOwned || this.topologyMetadata.namedTopologiesView().contains(taskId.topologyName())) continue;
            this.log.info("Cannot create the assigned task {} since it's topology name cannot be recognized, will put it aside as pending for now and create later when topology metadata gets refreshed", (Object)taskId);
            pendingTasks.put(taskId, entry.getValue());
            iter.remove();
        }
        return pendingTasks;
    }

    private Map<TaskId, RuntimeException> closeAndRecycleTasks(Map<Task, Set<TopicPartition>> tasksToRecycle, Set<Task> tasksToCloseClean) {
        String uncleanMessage;
        LinkedHashMap<TaskId, RuntimeException> taskCloseExceptions = new LinkedHashMap<TaskId, RuntimeException>();
        TreeSet<Task> tasksToCloseDirty = new TreeSet<Task>(Comparator.comparing(Task::id));
        ArrayList<Task> tasksToCheckpoint = new ArrayList<Task>(tasksToCloseClean);
        tasksToCheckpoint.addAll(tasksToRecycle.keySet());
        for (Task task : tasksToCheckpoint) {
            try {
                Map<TopicPartition, OffsetAndMetadata> offsets = task.prepareCommit();
                if (!offsets.isEmpty()) {
                    this.log.error("Task {} should have been committed when it was suspended, but it reports non-empty offsets {} to commit; this means it failed during last commit and hence should be closed dirty", (Object)task.id(), offsets);
                    tasksToCloseDirty.add(task);
                    continue;
                }
                if (task.isActive()) continue;
                task.suspend();
                task.postCommit(true);
            }
            catch (RuntimeException e) {
                uncleanMessage = String.format("Failed to checkpoint task %s. Attempting to close remaining tasks before re-throwing:", task.id());
                this.log.error(uncleanMessage, (Throwable)e);
                taskCloseExceptions.putIfAbsent(task.id(), e);
                tasksToCloseDirty.add(task);
            }
        }
        tasksToCloseClean.removeAll(tasksToCloseDirty);
        for (Task task : tasksToCloseClean) {
            try {
                this.closeTaskClean(task);
            }
            catch (RuntimeException closeTaskException) {
                uncleanMessage = String.format("Failed to close task %s cleanly. Attempting to close remaining tasks before re-throwing:", task.id());
                this.log.error(uncleanMessage, (Throwable)closeTaskException);
                if (task.state() != Task.State.CLOSED) {
                    tasksToCloseDirty.add(task);
                }
                taskCloseExceptions.putIfAbsent(task.id(), closeTaskException);
            }
        }
        tasksToRecycle.keySet().removeAll(tasksToCloseDirty);
        for (Map.Entry entry : tasksToRecycle.entrySet()) {
            Task oldTask = (Task)entry.getKey();
            Set inputPartitions = (Set)entry.getValue();
            try {
                if (oldTask.isActive()) {
                    StandbyTask standbyTask = this.convertActiveToStandby((StreamTask)oldTask, inputPartitions);
                    if (this.stateUpdater != null) {
                        this.tasks.removeTask(oldTask);
                        this.tasks.addPendingTasksToInit(Collections.singleton(standbyTask));
                        continue;
                    }
                    this.tasks.replaceActiveWithStandby(standbyTask);
                    continue;
                }
                StreamTask activeTask = this.convertStandbyToActive((StandbyTask)oldTask, inputPartitions);
                this.tasks.replaceStandbyWithActive(activeTask);
            }
            catch (RuntimeException e) {
                String uncleanMessage2 = String.format("Failed to recycle task %s cleanly. Attempting to close remaining tasks before re-throwing:", oldTask.id());
                this.log.error(uncleanMessage2, (Throwable)e);
                taskCloseExceptions.putIfAbsent(oldTask.id(), e);
                tasksToCloseDirty.add(oldTask);
            }
        }
        for (Task task : tasksToCloseDirty) {
            this.closeTaskDirty(task, true);
        }
        return taskCloseExceptions;
    }

    private StandbyTask convertActiveToStandby(StreamTask activeTask, Set<TopicPartition> partitions) {
        StandbyTask standbyTask = this.standbyTaskCreator.createStandbyTaskFromActive(activeTask, partitions);
        this.activeTaskCreator.closeAndRemoveTaskProducerIfNeeded(activeTask.id());
        return standbyTask;
    }

    private StreamTask convertStandbyToActive(StandbyTask standbyTask, Set<TopicPartition> partitions) {
        return this.activeTaskCreator.createActiveTaskFromStandby(standbyTask, partitions, this.mainConsumer);
    }

    boolean tryToCompleteRestoration(long now, java.util.function.Consumer<Set<TopicPartition>> offsetResetter) {
        boolean allRunning = true;
        this.changelogReader.enforceRestoreActive();
        LinkedList<Task> activeTasks = new LinkedList<Task>();
        for (Task task : this.tasks.allTasks()) {
            try {
                task.initializeIfNeeded();
                task.clearTaskTimeout();
            }
            catch (LockException lockException) {
                this.log.debug("Could not initialize task {} since: {}; will retry", (Object)task.id(), (Object)lockException.getMessage());
                allRunning = false;
            }
            catch (TimeoutException timeoutException) {
                task.maybeInitTaskTimeoutOrThrow(now, timeoutException);
                allRunning = false;
            }
            if (!task.isActive()) continue;
            activeTasks.add(task);
        }
        if (allRunning && !activeTasks.isEmpty()) {
            Set<TopicPartition> restored = this.changelogReader.completedChangelogs();
            for (Task task : activeTasks) {
                if (restored.containsAll(task.changelogPartitions())) {
                    try {
                        task.completeRestoration(offsetResetter);
                        task.clearTaskTimeout();
                    }
                    catch (TimeoutException timeoutException) {
                        task.maybeInitTaskTimeoutOrThrow(now, timeoutException);
                        this.log.debug(String.format("Could not complete restoration for %s due to the following exception; will retry", task.id()), (Throwable)timeoutException);
                        allRunning = false;
                    }
                    continue;
                }
                allRunning = false;
            }
        }
        if (allRunning) {
            this.mainConsumer.resume(this.mainConsumer.assignment());
            this.changelogReader.transitToUpdateStandby();
        }
        return allRunning;
    }

    public boolean checkStateUpdater(long now, java.util.function.Consumer<Set<TopicPartition>> offsetResetter) {
        this.addTasksToStateUpdater();
        if (this.stateUpdater.hasExceptionsAndFailedTasks()) {
            this.handleExceptionsFromStateUpdater();
        }
        if (this.stateUpdater.restoresActiveTasks()) {
            this.handleRestoredTasksFromStateUpdater(now, offsetResetter);
        }
        return !this.stateUpdater.restoresActiveTasks() && !this.tasks.hasPendingTasksToInit();
    }

    private void recycleTaskFromStateUpdater(Task task, Set<TopicPartition> inputPartitions, Set<Task> tasksToCloseDirty, Map<TaskId, RuntimeException> taskExceptions) {
        AbstractTask newTask = null;
        try {
            task.suspend();
            newTask = task.isActive() ? this.convertActiveToStandby((StreamTask)task, inputPartitions) : this.convertStandbyToActive((StandbyTask)task, inputPartitions);
            this.tasks.addPendingTasksToInit(Collections.singleton(newTask));
        }
        catch (RuntimeException e) {
            TaskId taskId = task.id();
            String uncleanMessage = String.format("Failed to recycle task %s cleanly. Attempting to close remaining tasks before re-throwing:", taskId);
            this.log.error(uncleanMessage, (Throwable)e);
            if (task.state() != Task.State.CLOSED) {
                tasksToCloseDirty.add(task);
            }
            if (newTask != null && newTask.state() != Task.State.CLOSED) {
                tasksToCloseDirty.add(newTask);
            }
            taskExceptions.putIfAbsent(taskId, e);
        }
    }

    private void closeTaskClean(Task task, Set<Task> tasksToCloseDirty, Map<TaskId, RuntimeException> taskExceptions) {
        try {
            task.suspend();
            task.closeClean();
            if (task.isActive()) {
                this.activeTaskCreator.closeAndRemoveTaskProducerIfNeeded(task.id());
            }
        }
        catch (RuntimeException e) {
            String uncleanMessage = String.format("Failed to close task %s cleanly. Attempting to close remaining tasks before re-throwing:", task.id());
            this.log.error(uncleanMessage, (Throwable)e);
            if (task.state() != Task.State.CLOSED) {
                tasksToCloseDirty.add(task);
            }
            taskExceptions.putIfAbsent(task.id(), e);
        }
    }

    private void transitRestoredTaskToRunning(Task task, long now, java.util.function.Consumer<Set<TopicPartition>> offsetResetter) {
        try {
            task.completeRestoration(offsetResetter);
            this.tasks.addTask(task);
            this.mainConsumer.resume(task.inputPartitions());
            task.clearTaskTimeout();
        }
        catch (TimeoutException timeoutException) {
            task.maybeInitTaskTimeoutOrThrow(now, timeoutException);
            this.stateUpdater.add(task);
            this.log.debug(String.format("Could not complete restoration for %s due to the following exception; adding the task back to the state updater and will retry", task.id()), (Throwable)timeoutException);
        }
    }

    private void addTasksToStateUpdater() {
        LinkedHashMap<TaskId, RuntimeException> taskExceptions = new LinkedHashMap<TaskId, RuntimeException>();
        for (Task task : this.tasks.drainPendingTasksToInit()) {
            try {
                this.addTaskToStateUpdater(task);
            }
            catch (RuntimeException e) {
                this.tasks.addFailedTask(task);
                taskExceptions.put(task.id(), e);
            }
        }
        this.maybeThrowTaskExceptions(taskExceptions);
    }

    private void addTaskToStateUpdater(Task task) {
        try {
            task.initializeIfNeeded();
            this.stateUpdater.add(task);
        }
        catch (LockException lockException) {
            this.log.info("Encountered lock exception. Reattempting locking the state in the next iteration.", (Throwable)lockException);
            this.tasks.addPendingTasksToInit(Collections.singleton(task));
        }
    }

    public void handleExceptionsFromStateUpdater() {
        Map<TaskId, RuntimeException> taskExceptions = this.collectExceptionsAndFailedTasksFromStateUpdater();
        this.maybeThrowTaskExceptions(taskExceptions);
    }

    public Map<TaskId, RuntimeException> collectExceptionsAndFailedTasksFromStateUpdater() {
        LinkedHashMap<TaskId, RuntimeException> taskExceptions = new LinkedHashMap<TaskId, RuntimeException>();
        for (StateUpdater.ExceptionAndTask exceptionAndTask : this.stateUpdater.drainExceptionsAndFailedTasks()) {
            RuntimeException exception = exceptionAndTask.exception();
            Task failedTask = exceptionAndTask.task();
            this.tasks.addFailedTask(failedTask);
            taskExceptions.put(failedTask.id(), exception);
        }
        return taskExceptions;
    }

    private void handleRestoredTasksFromStateUpdater(long now, java.util.function.Consumer<Set<TopicPartition>> offsetResetter) {
        Duration timeout = Duration.ZERO;
        for (Task task : this.stateUpdater.drainRestoredActiveTasks(timeout)) {
            this.transitRestoredTaskToRunning(task, now, offsetResetter);
        }
    }

    void handleRevocation(Collection<TopicPartition> revokedPartitions) {
        boolean shouldCommitAdditionalTasks;
        HashSet<TopicPartition> remainingRevokedPartitions = new HashSet<TopicPartition>(revokedPartitions);
        HashSet<Task> revokedActiveTasks = new HashSet<Task>();
        HashSet<Task> commitNeededActiveTasks = new HashSet<Task>();
        HashMap<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsPerTask = new HashMap<Task, Map<TopicPartition, OffsetAndMetadata>>();
        AtomicReference<Object> firstException = new AtomicReference<Object>(null);
        Set<TaskId> lockedTaskIds = this.activeRunningTaskIterable().stream().map(Task::id).collect(Collectors.toSet());
        this.maybeLockTasks(lockedTaskIds);
        for (Task task : this.activeRunningTaskIterable()) {
            if (remainingRevokedPartitions.containsAll(task.inputPartitions())) {
                revokedActiveTasks.add(task);
                remainingRevokedPartitions.removeAll(task.inputPartitions());
                continue;
            }
            if (!task.commitNeeded()) continue;
            commitNeededActiveTasks.add(task);
        }
        this.revokeTasksInStateUpdater(remainingRevokedPartitions);
        if (!remainingRevokedPartitions.isEmpty()) {
            this.log.debug("The following revoked partitions {} are missing from the current task partitions. It could potentially be due to race condition of consumer detecting the heartbeat failure, or the tasks have been cleaned up by the handleAssignment callback.", remainingRevokedPartitions);
        }
        this.prepareCommitAndAddOffsetsToMap(revokedActiveTasks, consumedOffsetsPerTask);
        boolean bl = shouldCommitAdditionalTasks = !consumedOffsetsPerTask.isEmpty();
        if (shouldCommitAdditionalTasks) {
            this.prepareCommitAndAddOffsetsToMap(commitNeededActiveTasks, consumedOffsetsPerTask);
        }
        HashSet<Task> dirtyTasks = new HashSet<Task>();
        try {
            this.taskExecutor.commitOffsetsOrTransaction(consumedOffsetsPerTask);
        }
        catch (TaskCorruptedException e) {
            this.log.warn("Some tasks were corrupted when trying to commit offsets, these will be cleaned and revived: {}", e.corruptedTasks());
            dirtyTasks.addAll(this.tasks.tasks(e.corruptedTasks()));
            this.closeDirtyAndRevive(dirtyTasks, true);
        }
        catch (TimeoutException e) {
            this.log.warn("Timed out while trying to commit all tasks during revocation, these will be cleaned and revived");
            dirtyTasks.addAll(consumedOffsetsPerTask.keySet());
            this.closeDirtyAndRevive(dirtyTasks, false);
        }
        catch (RuntimeException e) {
            this.log.error("Exception caught while committing those revoked tasks " + revokedActiveTasks, (Throwable)e);
            firstException.compareAndSet(null, e);
            dirtyTasks.addAll(consumedOffsetsPerTask.keySet());
        }
        for (Task task : revokedActiveTasks) {
            if (dirtyTasks.contains(task)) continue;
            try {
                task.postCommit(true);
            }
            catch (RuntimeException e) {
                this.log.error("Exception caught while post-committing task " + task.id(), (Throwable)e);
                this.maybeSetFirstException(false, this.maybeWrapTaskException(e, task.id()), firstException);
            }
        }
        if (shouldCommitAdditionalTasks) {
            for (Task task : commitNeededActiveTasks) {
                if (dirtyTasks.contains(task)) continue;
                try {
                    task.postCommit(false);
                }
                catch (RuntimeException e) {
                    this.log.error("Exception caught while post-committing task " + task.id(), (Throwable)e);
                    this.maybeSetFirstException(false, this.maybeWrapTaskException(e, task.id()), firstException);
                }
            }
        }
        for (Task task : revokedActiveTasks) {
            try {
                task.suspend();
            }
            catch (RuntimeException e) {
                this.log.error("Caught the following exception while trying to suspend revoked task " + task.id(), (Throwable)e);
                this.maybeSetFirstException(false, this.maybeWrapTaskException(e, task.id()), firstException);
            }
        }
        this.maybeUnlockTasks(lockedTaskIds);
        if (firstException.get() != null) {
            throw (RuntimeException)firstException.get();
        }
    }

    private void revokeTasksInStateUpdater(Set<TopicPartition> remainingRevokedPartitions) {
        if (this.stateUpdater != null) {
            LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
            HashMap<TaskId, RuntimeException> failedTasksFromStateUpdater = new HashMap<TaskId, RuntimeException>();
            for (Task restoringTask : this.stateUpdater.getTasks()) {
                if (!restoringTask.isActive() || !remainingRevokedPartitions.containsAll(restoringTask.inputPartitions())) continue;
                futures.put(restoringTask.id(), this.stateUpdater.remove(restoringTask.id()));
                remainingRevokedPartitions.removeAll(restoringTask.inputPartitions());
            }
            this.getNonFailedTasks(futures, failedTasksFromStateUpdater).forEach(task -> {
                task.suspend();
                this.tasks.addTask((Task)task);
            });
            this.maybeThrowTaskExceptions(failedTasksFromStateUpdater);
        }
    }

    private void prepareCommitAndAddOffsetsToMap(Set<Task> tasksToPrepare, Map<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsPerTask) {
        for (Task task : tasksToPrepare) {
            try {
                Map<TopicPartition, OffsetAndMetadata> committableOffsets = task.prepareCommit();
                if (committableOffsets.isEmpty()) continue;
                consumedOffsetsPerTask.put(task, committableOffsets);
            }
            catch (StreamsException e) {
                e.setTaskId(task.id());
                throw e;
            }
            catch (Exception e) {
                throw new StreamsException(e, task.id());
            }
        }
    }

    void handleLostAll() {
        this.log.debug("Closing lost active tasks as zombies.");
        this.closeRunningTasksDirty();
        this.removeLostActiveTasksFromStateUpdaterAndPendingTasksToInit();
        if (this.processingMode == StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_V2) {
            this.activeTaskCreator.reInitializeThreadProducer();
        }
    }

    private void closeRunningTasksDirty() {
        Set<Task> allTask = this.tasks.allTasks();
        Set<TaskId> allTaskIds = this.tasks.allTaskIds();
        this.maybeLockTasks(allTaskIds);
        for (Task task : allTask) {
            if (!task.isActive()) continue;
            this.closeTaskDirty(task, true);
        }
        this.maybeUnlockTasks(allTaskIds);
    }

    private void removeLostActiveTasksFromStateUpdaterAndPendingTasksToInit() {
        if (this.stateUpdater != null) {
            LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
            HashMap<TaskId, RuntimeException> failedTasksDuringCleanClose = new HashMap<TaskId, RuntimeException>();
            HashSet<Task> tasksToCloseClean = new HashSet<Task>(this.tasks.drainPendingActiveTasksToInit());
            HashSet<Task> tasksToCloseDirty = new HashSet<Task>();
            for (Task restoringTask : this.stateUpdater.getTasks()) {
                if (!restoringTask.isActive()) continue;
                futures.put(restoringTask.id(), this.stateUpdater.remove(restoringTask.id()));
            }
            this.addToTasksToClose(futures, tasksToCloseClean, tasksToCloseDirty);
            for (Task task : tasksToCloseClean) {
                this.closeTaskClean(task, tasksToCloseDirty, failedTasksDuringCleanClose);
            }
            for (Task task : tasksToCloseDirty) {
                this.closeTaskDirty(task, false);
            }
        }
    }

    public void signalResume() {
        if (this.stateUpdater != null) {
            this.stateUpdater.signalResume();
        }
        if (this.schedulingTaskManager != null) {
            this.schedulingTaskManager.signalTaskExecutors();
        }
    }

    public Map<TaskId, Long> getTaskOffsetSums() {
        HashMap<TaskId, Long> taskOffsetSums = new HashMap<TaskId, Long>();
        Map<TaskId, Task> tasks = this.allTasks();
        Set<TaskId> lockedTaskDirectoriesOfNonOwnedTasksAndClosedAndCreatedTasks = Utils.union(HashSet::new, this.lockedTaskDirectories, tasks.keySet());
        for (Task task : tasks.values()) {
            if (task.state() == Task.State.CREATED || task.state() == Task.State.CLOSED) continue;
            Map<TopicPartition, Long> changelogOffsets = task.changelogOffsets();
            if (changelogOffsets.isEmpty()) {
                this.log.debug("Skipping to encode apparently stateless (or non-logged) offset sum for task {}", (Object)task.id());
            } else {
                taskOffsetSums.put(task.id(), this.sumOfChangelogOffsets(task.id(), changelogOffsets));
            }
            lockedTaskDirectoriesOfNonOwnedTasksAndClosedAndCreatedTasks.remove(task.id());
        }
        for (TaskId id : lockedTaskDirectoriesOfNonOwnedTasksAndClosedAndCreatedTasks) {
            File checkpointFile = this.stateDirectory.checkpointFileFor(id);
            try {
                if (!checkpointFile.exists()) continue;
                taskOffsetSums.put(id, this.sumOfChangelogOffsets(id, new OffsetCheckpoint(checkpointFile).read()));
            }
            catch (IOException e) {
                this.log.warn(String.format("Exception caught while trying to read checkpoint for task %s:", id), (Throwable)e);
            }
        }
        return taskOffsetSums;
    }

    private void tryToLockAllNonEmptyTaskDirectories() {
        this.lockedTaskDirectories.clear();
        Map<TaskId, Task> allTasks = this.allTasks();
        for (StateDirectory.TaskDirectory taskDir : this.stateDirectory.listNonEmptyTaskDirectories()) {
            File dir = taskDir.file();
            String namedTopology = taskDir.namedTopology();
            try {
                TaskId id = StateManagerUtil.parseTaskDirectoryName(dir.getName(), namedTopology);
                if (!this.stateDirectory.lock(id)) continue;
                if (this.stateDirectory.directoryForTaskIsEmpty(id)) {
                    this.log.debug("Releasing lock on empty directory for task {}", (Object)id);
                    this.stateDirectory.unlock(id);
                    continue;
                }
                this.lockedTaskDirectories.add(id);
                if (allTasks.containsKey(id)) continue;
                this.log.debug("Temporarily locked unassigned task {} for the upcoming rebalance", (Object)id);
            }
            catch (TaskIdFormatException taskIdFormatException) {}
        }
    }

    private void releaseLockedDirectoriesForTasks(Set<TaskId> tasksToUnlock) {
        Iterator<TaskId> taskIdIterator = this.lockedTaskDirectories.iterator();
        while (taskIdIterator.hasNext()) {
            TaskId id = taskIdIterator.next();
            if (!tasksToUnlock.contains(id)) continue;
            this.stateDirectory.unlock(id);
            taskIdIterator.remove();
        }
    }

    private void releaseLockedUnassignedTaskDirectories() {
        Iterator<TaskId> taskIdIterator = this.lockedTaskDirectories.iterator();
        Map<TaskId, Task> allTasks = this.allTasks();
        while (taskIdIterator.hasNext()) {
            TaskId id = taskIdIterator.next();
            if (allTasks.containsKey(id)) continue;
            this.stateDirectory.unlock(id);
            taskIdIterator.remove();
        }
    }

    private long sumOfChangelogOffsets(TaskId id, Map<TopicPartition, Long> changelogOffsets) {
        long offsetSum = 0L;
        for (Map.Entry<TopicPartition, Long> changelogEntry : changelogOffsets.entrySet()) {
            long offset = changelogEntry.getValue();
            if (offset == -2L) {
                return -2L;
            }
            if (offset == -4L) continue;
            if (offset < 0L) {
                throw new StreamsException(new IllegalStateException("Expected not to get a sentinel offset, but got: " + changelogEntry), id);
            }
            if ((offsetSum += offset) >= 0L) continue;
            this.log.warn("Sum of changelog offsets for task {} overflowed, pinning to Long.MAX_VALUE", (Object)id);
            return Long.MAX_VALUE;
        }
        return offsetSum;
    }

    private void closeTaskDirty(Task task, boolean removeFromTasksRegistry) {
        try {
            task.prepareCommit();
        }
        catch (RuntimeException swallow) {
            this.log.error("Error flushing caches of dirty task {}", (Object)task.id(), (Object)swallow);
        }
        try {
            task.suspend();
        }
        catch (RuntimeException swallow) {
            this.log.error("Error suspending dirty task {}: {}", (Object)task.id(), (Object)swallow.getMessage());
        }
        task.closeDirty();
        try {
            if (removeFromTasksRegistry) {
                this.tasks.removeTask(task);
            }
            if (task.isActive()) {
                this.activeTaskCreator.closeAndRemoveTaskProducerIfNeeded(task.id());
            }
        }
        catch (RuntimeException swallow) {
            this.log.error("Error removing dirty task {}: {}", (Object)task.id(), (Object)swallow.getMessage());
        }
    }

    private void closeTaskClean(Task task) {
        task.closeClean();
        this.tasks.removeTask(task);
        if (task.isActive()) {
            this.activeTaskCreator.closeAndRemoveTaskProducerIfNeeded(task.id());
        }
    }

    void shutdown(boolean clean) {
        this.shutdownStateUpdater();
        this.shutdownSchedulingTaskManager();
        AtomicReference<Object> firstException = new AtomicReference<Object>(null);
        TreeSet<Task> activeTasks = new TreeSet<Task>(Comparator.comparing(Task::id));
        activeTasks.addAll(this.tasks.activeTasks());
        TaskManager.executeAndMaybeSwallow(clean, () -> this.closeAndCleanUpTasks(activeTasks, this.standbyTaskIterable(), clean), (RuntimeException e) -> firstException.compareAndSet(null, e), (RuntimeException e) -> this.log.warn("Ignoring an exception while unlocking remaining task directories.", (Throwable)e));
        TaskManager.executeAndMaybeSwallow(clean, this.activeTaskCreator::closeThreadProducerIfNeeded, (RuntimeException e) -> firstException.compareAndSet(null, e), (RuntimeException e) -> this.log.warn("Ignoring an exception while closing thread producer.", (Throwable)e));
        this.tasks.clear();
        TaskManager.executeAndMaybeSwallow(clean, this::releaseLockedUnassignedTaskDirectories, (RuntimeException e) -> firstException.compareAndSet(null, e), (RuntimeException e) -> this.log.warn("Ignoring an exception while unlocking remaining task directories.", (Throwable)e));
        RuntimeException fatalException = firstException.get();
        if (fatalException != null) {
            throw fatalException;
        }
        this.log.info("Shutdown complete");
    }

    private void shutdownStateUpdater() {
        if (this.stateUpdater != null) {
            LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>> futures = new LinkedHashMap<TaskId, CompletableFuture<StateUpdater.RemovedTaskResult>>();
            for (Task task : this.stateUpdater.getTasks()) {
                CompletableFuture<StateUpdater.RemovedTaskResult> future = this.stateUpdater.remove(task.id());
                futures.put(task.id(), future);
            }
            HashSet<Task> tasksToCloseClean = new HashSet<Task>();
            HashSet<Task> tasksToCloseDirty = new HashSet<Task>();
            this.addToTasksToClose(futures, tasksToCloseClean, tasksToCloseDirty);
            this.stateUpdater.shutdown(Duration.ofMillis(Long.MAX_VALUE));
            for (Task task : tasksToCloseClean) {
                this.tasks.addTask(task);
            }
            for (Task task : tasksToCloseDirty) {
                this.closeTaskDirty(task, false);
            }
            for (StateUpdater.ExceptionAndTask exceptionAndTask : this.stateUpdater.drainExceptionsAndFailedTasks()) {
                Task failedTask = exceptionAndTask.task();
                this.closeTaskDirty(failedTask, false);
            }
        }
    }

    private void shutdownSchedulingTaskManager() {
        if (this.schedulingTaskManager != null) {
            this.schedulingTaskManager.shutdown(Duration.ofMillis(Long.MAX_VALUE));
        }
    }

    void closeAndCleanUpTasks(Collection<Task> activeTasks, Collection<Task> standbyTasks, boolean clean) {
        AtomicReference<Object> firstException = new AtomicReference<Object>(null);
        Set<TaskId> ids = activeTasks.stream().map(Task::id).collect(Collectors.toSet());
        this.maybeLockTasks(ids);
        HashSet<Task> tasksToCloseDirty = new HashSet<Task>();
        tasksToCloseDirty.addAll(this.tryCloseCleanActiveTasks(activeTasks, clean, firstException));
        tasksToCloseDirty.addAll(this.tryCloseCleanStandbyTasks(standbyTasks, clean, firstException));
        for (Task task : tasksToCloseDirty) {
            this.closeTaskDirty(task, true);
        }
        this.maybeUnlockTasks(ids);
        RuntimeException exception = firstException.get();
        if (exception != null) {
            throw exception;
        }
    }

    private Collection<Task> tryCloseCleanActiveTasks(Collection<Task> activeTasksToClose, boolean clean, AtomicReference<RuntimeException> firstException) {
        if (!clean) {
            return this.activeTaskIterable();
        }
        Comparator<Task> byId = Comparator.comparing(Task::id);
        TreeSet<Task> tasksToCommit = new TreeSet<Task>(byId);
        TreeSet<Task> tasksToCloseDirty = new TreeSet<Task>(byId);
        TreeSet<Task> tasksToCloseClean = new TreeSet<Task>(byId);
        HashMap<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsAndMetadataPerTask = new HashMap<Task, Map<TopicPartition, OffsetAndMetadata>>();
        for (Task task2 : activeTasksToClose) {
            try {
                Map<TopicPartition, OffsetAndMetadata> committableOffsets = task2.prepareCommit();
                tasksToCommit.add(task2);
                if (!committableOffsets.isEmpty()) {
                    consumedOffsetsAndMetadataPerTask.put(task2, committableOffsets);
                }
                tasksToCloseClean.add(task2);
            }
            catch (TaskMigratedException e) {
                tasksToCloseDirty.add(task2);
            }
            catch (StreamsException e) {
                e.setTaskId(task2.id());
                firstException.compareAndSet(null, e);
                tasksToCloseDirty.add(task2);
            }
            catch (RuntimeException e) {
                firstException.compareAndSet(null, new StreamsException(e, task2.id()));
                tasksToCloseDirty.add(task2);
            }
        }
        if (this.processingMode == StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_V2 && !tasksToCloseDirty.isEmpty()) {
            tasksToCloseClean.removeAll(tasksToCommit);
            tasksToCloseDirty.addAll(tasksToCommit);
        } else {
            try {
                this.taskExecutor.commitOffsetsOrTransaction(consumedOffsetsAndMetadataPerTask);
            }
            catch (RuntimeException e) {
                this.log.error("Exception caught while committing tasks " + consumedOffsetsAndMetadataPerTask.keySet(), (Throwable)e);
                this.maybeSetFirstException(false, e, firstException);
                if (e instanceof TaskCorruptedException) {
                    TaskCorruptedException taskCorruptedException = (TaskCorruptedException)e;
                    Set<TaskId> corruptedTaskIds = taskCorruptedException.corruptedTasks();
                    Set corruptedTasks = tasksToCommit.stream().filter(task -> corruptedTaskIds.contains(task.id())).collect(Collectors.toSet());
                    tasksToCloseClean.removeAll(corruptedTasks);
                    tasksToCloseDirty.addAll(corruptedTasks);
                }
                tasksToCloseClean.removeAll(tasksToCommit);
                tasksToCloseDirty.addAll(tasksToCommit);
            }
            for (Task task2 : this.activeTaskIterable()) {
                try {
                    task2.postCommit(true);
                }
                catch (RuntimeException e) {
                    this.log.error("Exception caught while post-committing task " + task2.id(), (Throwable)e);
                    this.maybeSetFirstException(false, this.maybeWrapTaskException(e, task2.id()), firstException);
                    tasksToCloseDirty.add(task2);
                    tasksToCloseClean.remove(task2);
                }
            }
        }
        for (Task task2 : tasksToCloseClean) {
            try {
                task2.suspend();
                this.closeTaskClean(task2);
            }
            catch (RuntimeException e) {
                this.log.error("Exception caught while clean-closing active task {}: {}", (Object)task2.id(), (Object)e.getMessage());
                if (task2.state() != Task.State.CLOSED) {
                    tasksToCloseDirty.add(task2);
                }
                this.maybeSetFirstException(true, this.maybeWrapTaskException(e, task2.id()), firstException);
            }
        }
        return tasksToCloseDirty;
    }

    private Collection<Task> tryCloseCleanStandbyTasks(Collection<Task> standbyTasksToClose, boolean clean, AtomicReference<RuntimeException> firstException) {
        if (!clean) {
            return this.standbyTaskIterable();
        }
        HashSet<Task> tasksToCloseDirty = new HashSet<Task>();
        for (Task task : standbyTasksToClose) {
            try {
                task.prepareCommit();
                task.postCommit(true);
                task.suspend();
                this.closeTaskClean(task);
            }
            catch (RuntimeException e) {
                this.log.error("Exception caught while clean-closing standby task {}: {}", (Object)task.id(), (Object)e.getMessage());
                if (task.state() != Task.State.CLOSED) {
                    tasksToCloseDirty.add(task);
                }
                this.maybeSetFirstException(true, this.maybeWrapTaskException(e, task.id()), firstException);
            }
        }
        return tasksToCloseDirty;
    }

    Set<TaskId> activeTaskIds() {
        return this.activeTaskStream().map(Task::id).collect(Collectors.toSet());
    }

    Set<TaskId> activeRunningTaskIds() {
        return this.activeRunningTaskStream().map(Task::id).collect(Collectors.toSet());
    }

    Set<TaskId> standbyTaskIds() {
        return this.standbyTaskStream().map(Task::id).collect(Collectors.toSet());
    }

    Map<TaskId, Task> allTasks() {
        if (this.stateUpdater != null) {
            Map<TaskId, Task> ret = this.stateUpdater.getTasks().stream().collect(Collectors.toMap(Task::id, x -> x));
            ret.putAll(this.tasks.allTasksPerId());
            ret.putAll(this.tasks.pendingTasksToInit().stream().collect(Collectors.toMap(Task::id, x -> x)));
            return ret;
        }
        return this.tasks.allTasksPerId();
    }

    Map<TaskId, Task> allOwnedTasks() {
        return this.tasks.allTasksPerId();
    }

    Set<Task> readOnlyAllTasks() {
        if (this.stateUpdater != null) {
            HashSet<Task> ret = new HashSet<Task>(this.stateUpdater.getTasks());
            ret.addAll(this.tasks.allTasks());
            return Collections.unmodifiableSet(ret);
        }
        return Collections.unmodifiableSet(this.tasks.allTasks());
    }

    Map<TaskId, Task> notPausedTasks() {
        return Collections.unmodifiableMap(this.tasks.allTasks().stream().filter(t -> !this.topologyMetadata.isPaused(t.id().topologyName())).collect(Collectors.toMap(Task::id, v -> v)));
    }

    Map<TaskId, Task> activeTaskMap() {
        return this.activeTaskStream().collect(Collectors.toMap(Task::id, t -> t));
    }

    List<Task> activeTaskIterable() {
        return this.activeTaskStream().collect(Collectors.toList());
    }

    List<Task> activeRunningTaskIterable() {
        return this.activeRunningTaskStream().collect(Collectors.toList());
    }

    private Stream<Task> activeTaskStream() {
        if (this.stateUpdater != null) {
            return Stream.concat(this.activeRunningTaskStream(), this.stateUpdater.getTasks().stream().filter(Task::isActive));
        }
        return this.activeRunningTaskStream();
    }

    private Stream<Task> activeRunningTaskStream() {
        return this.tasks.allTasks().stream().filter(Task::isActive);
    }

    Map<TaskId, Task> standbyTaskMap() {
        return this.standbyTaskStream().collect(Collectors.toMap(Task::id, t -> t));
    }

    private List<Task> standbyTaskIterable() {
        return this.standbyTaskStream().collect(Collectors.toList());
    }

    private Stream<Task> standbyTaskStream() {
        Stream<Task> standbyTasksInTaskRegistry = this.tasks.allTasks().stream().filter(t -> !t.isActive());
        if (this.stateUpdater != null) {
            return Stream.concat(this.stateUpdater.getStandbyTasks().stream(), standbyTasksInTaskRegistry);
        }
        return standbyTasksInTaskRegistry;
    }

    int commitAll() {
        return this.commit(this.tasks.allTasks());
    }

    public void resumePollingForPartitionsWithAvailableSpace() {
        for (Task t : this.tasks.activeTasks()) {
            t.resumePollingForPartitionsWithAvailableSpace();
        }
    }

    public void updateLags() {
        for (Task t : this.tasks.activeTasks()) {
            t.updateLags();
        }
    }

    public void signalTaskExecutors() {
        if (this.schedulingTaskManager != null) {
            this.schedulingTaskManager.signalTaskExecutors();
        }
    }

    void addRecordsToTasks(ConsumerRecords<byte[], byte[]> records) {
        for (TopicPartition partition : records.partitions()) {
            Task activeTask = this.tasks.activeTasksForInputPartition(partition);
            if (activeTask == null) {
                this.log.error("Unable to locate active task for received-record partition {}. Current tasks: {}", (Object)partition, (Object)this.toString(">"));
                throw new NullPointerException("Task was unexpectedly missing for partition " + partition);
            }
            activeTask.addRecords(partition, records.records(partition));
        }
    }

    private void maybeLockTasks(Set<TaskId> ids) {
        if (this.schedulingTaskManager != null && !ids.isEmpty()) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Locking tasks {}", (Object)ids.stream().map(TaskId::toString).collect(Collectors.joining(", ")));
            }
            boolean locked = false;
            while (!locked) {
                try {
                    this.schedulingTaskManager.lockTasks(ids).get();
                    locked = true;
                }
                catch (InterruptedException e) {
                    this.log.warn("Interrupted while waiting for tasks {} to be locked", (Object)ids.stream().map(TaskId::toString).collect(Collectors.joining(",")));
                }
                catch (ExecutionException e) {
                    this.log.info("Failed to lock tasks");
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private void maybeUnlockTasks(Set<TaskId> ids) {
        if (this.schedulingTaskManager != null && !ids.isEmpty()) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Unlocking tasks {}", (Object)ids.stream().map(TaskId::toString).collect(Collectors.joining(", ")));
            }
            this.schedulingTaskManager.unlockTasks(ids);
        }
    }

    public void maybeThrowTaskExceptionsFromProcessingThreads() {
        if (this.schedulingTaskManager != null) {
            this.maybeThrowTaskExceptions(this.schedulingTaskManager.drainUncaughtExceptions());
        }
    }

    int commit(Collection<Task> tasksToCommit) {
        int committed = 0;
        Set<TaskId> ids = tasksToCommit.stream().map(Task::id).collect(Collectors.toSet());
        this.maybeLockTasks(ids);
        this.maybeThrowTaskExceptionsFromProcessingThreads();
        HashMap<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsAndMetadataPerTask = new HashMap<Task, Map<TopicPartition, OffsetAndMetadata>>();
        try {
            committed = this.commitTasksAndMaybeUpdateCommittableOffsets(tasksToCommit, consumedOffsetsAndMetadataPerTask);
        }
        catch (TimeoutException timeoutException) {
            consumedOffsetsAndMetadataPerTask.keySet().forEach(t -> t.maybeInitTaskTimeoutOrThrow(this.time.milliseconds(), timeoutException));
        }
        this.maybeUnlockTasks(ids);
        return committed;
    }

    int maybeCommitActiveTasksPerUserRequested() {
        if (this.rebalanceInProgress) {
            return -1;
        }
        for (Task task : this.activeRunningTaskIterable()) {
            if (!task.commitRequested() || !task.commitNeeded()) continue;
            return this.commit(this.activeRunningTaskIterable());
        }
        return 0;
    }

    private int commitTasksAndMaybeUpdateCommittableOffsets(Collection<Task> tasksToCommit, Map<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsAndMetadata) {
        if (this.rebalanceInProgress) {
            return -1;
        }
        return this.taskExecutor.commitTasksAndMaybeUpdateCommittableOffsets(tasksToCommit, consumedOffsetsAndMetadata);
    }

    public void updateTaskEndMetadata(TopicPartition topicPartition, Long offset) {
        for (Task task : this.tasks.activeTasks()) {
            if (!(task instanceof StreamTask) || !task.inputPartitions().contains(topicPartition)) continue;
            ((StreamTask)task).updateEndOffsets(topicPartition, offset);
        }
    }

    void handleTopologyUpdates() {
        this.topologyMetadata.executeTopologyUpdatesAndBumpThreadVersion(this::createPendingTasks, this::maybeCloseTasksFromRemovedTopologies);
        if (this.topologyMetadata.isEmpty()) {
            this.log.info("Proactively unsubscribing from all topics due to empty topology");
            this.mainConsumer.unsubscribe();
        }
        this.topologyMetadata.maybeNotifyTopologyVersionListeners();
    }

    void maybeCloseTasksFromRemovedTopologies(Set<String> currentNamedTopologies) {
        try {
            HashSet<Task> activeTasksToRemove = new HashSet<Task>();
            HashSet<Task> standbyTasksToRemove = new HashSet<Task>();
            for (Task task : this.tasks.allTasks()) {
                if (currentNamedTopologies.contains(task.id().topologyName())) continue;
                if (task.isActive()) {
                    activeTasksToRemove.add(task);
                    continue;
                }
                standbyTasksToRemove.add(task);
            }
            Set allTasksToRemove = Utils.union(HashSet::new, activeTasksToRemove, standbyTasksToRemove);
            this.closeAndCleanUpTasks(activeTasksToRemove, standbyTasksToRemove, true);
            this.releaseLockedDirectoriesForTasks(allTasksToRemove.stream().map(Task::id).collect(Collectors.toSet()));
        }
        catch (Exception e) {
            this.log.error("Caught the following exception while closing tasks from a removed topology:", (Throwable)e);
        }
    }

    void createPendingTasks(Set<String> currentNamedTopologies) {
        Map<TaskId, Set<TopicPartition>> activeTasksToCreate = this.tasks.drainPendingActiveTasksForTopologies(currentNamedTopologies);
        Map<TaskId, Set<TopicPartition>> standbyTasksToCreate = this.tasks.drainPendingStandbyTasksForTopologies(currentNamedTopologies);
        this.createNewTasks(activeTasksToCreate, standbyTasksToCreate);
    }

    int process(int maxNumRecords, Time time) {
        return this.taskExecutor.process(maxNumRecords, time);
    }

    void recordTaskProcessRatio(long totalProcessLatencyMs, long now) {
        for (Task task : this.activeRunningTaskIterable()) {
            task.recordProcessTimeRatioAndBufferSize(totalProcessLatencyMs, now);
        }
    }

    int punctuate() {
        return this.taskExecutor.punctuate();
    }

    void maybePurgeCommittedRecords() {
        if (this.deleteRecordsResult == null || this.deleteRecordsResult.all().isDone()) {
            if (this.deleteRecordsResult != null && this.deleteRecordsResult.all().isCompletedExceptionally()) {
                this.log.debug("Previous delete-records request has failed: {}. Try sending the new request now", this.deleteRecordsResult.lowWatermarks());
            }
            HashMap<TopicPartition, RecordsToDelete> recordsToDelete = new HashMap<TopicPartition, RecordsToDelete>();
            for (Task task : this.activeRunningTaskIterable()) {
                for (Map.Entry<TopicPartition, Long> entry : task.purgeableOffsets().entrySet()) {
                    recordsToDelete.put(entry.getKey(), RecordsToDelete.beforeOffset(entry.getValue()));
                }
            }
            if (!recordsToDelete.isEmpty()) {
                this.deleteRecordsResult = this.adminClient.deleteRecords(recordsToDelete);
                this.log.trace("Sent delete-records request: {}", recordsToDelete);
            }
        }
    }

    public String toString() {
        return this.toString("");
    }

    public String toString(String indent) {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("TaskManager\n");
        stringBuilder.append(indent).append("\tMetadataState:\n");
        stringBuilder.append(indent).append("\tTasks:\n");
        for (Task task : this.tasks.allTasks()) {
            stringBuilder.append(indent).append("\t\t").append(task.id()).append(" ").append((Object)task.state()).append(" ").append(task.getClass().getSimpleName()).append('(').append(task.isActive() ? "active" : "standby").append(')');
        }
        return stringBuilder.toString();
    }

    Map<MetricName, Metric> producerMetrics() {
        return this.activeTaskCreator.producerMetrics();
    }

    Set<String> producerClientIds() {
        return this.activeTaskCreator.producerClientIds();
    }

    Set<TaskId> lockedTaskDirectories() {
        return Collections.unmodifiableSet(this.lockedTaskDirectories);
    }

    private void maybeSetFirstException(boolean ignoreTaskMigrated, RuntimeException exception, AtomicReference<RuntimeException> firstException) {
        if (!ignoreTaskMigrated || !(exception instanceof TaskMigratedException)) {
            firstException.compareAndSet(null, exception);
        }
    }

    private StreamsException maybeWrapTaskException(RuntimeException exception, TaskId taskId) {
        if (exception instanceof StreamsException) {
            StreamsException streamsException = (StreamsException)exception;
            streamsException.setTaskId(taskId);
            return streamsException;
        }
        return new StreamsException(exception, taskId);
    }

    public static void executeAndMaybeSwallow(boolean clean, Runnable runnable, java.util.function.Consumer<RuntimeException> actionIfClean, java.util.function.Consumer<RuntimeException> actionIfNotClean) {
        try {
            runnable.run();
        }
        catch (RuntimeException e) {
            if (clean) {
                actionIfClean.accept(e);
            }
            actionIfNotClean.accept(e);
        }
    }

    public static void executeAndMaybeSwallow(boolean clean, Runnable runnable, String name, Logger log) {
        TaskManager.executeAndMaybeSwallow(clean, runnable, (RuntimeException e) -> {
            throw e;
        }, (RuntimeException e) -> log.debug("Ignoring error in unclean {}", (Object)name));
    }

    boolean needsInitializationOrRestoration() {
        return this.activeTaskStream().anyMatch(Task::needsInitializationOrRestoration);
    }

    void addTask(Task task) {
        this.tasks.addTask(task);
    }
}

