/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.core.container.service.internal;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.internal.IAccessControlList;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IAclStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;

public class AclService implements IAccessControlList {
	private final IAclStore aclStore;
	private final Supplier<Optional<AuditLogService<AccessControlEntry, AccessControlEntry>>> auditLogSupplier;
	private final Container container;

	public AclService(BmContext ctx, SecurityContext sc, BaseContainerDescriptor desc) {
		this.container = Container.create(desc.uid, desc.type, desc.name, desc.owner, desc.domainUid,
				desc.defaultContainer);
		container.id = desc.internalId;
		aclStore = RepositoryProvider.instance(IAclStore.class, ctx, ContainerUid.of(desc.uid));
		auditLogSupplier = () -> {
			if (StateContext.getState().equals(SystemState.CORE_STATE_RUNNING)) {
				var auditLog = new ValueAuditLogService<>(ctx.su(sc), desc, new AccessControlEntryAuditLogMapper(desc));
				auditLog.setType("containeracl");
				return Optional.of(auditLog);
			} else {
				return Optional.empty();
			}
		};
	}

	public AclService(BmContext ctx, SecurityContext sc, Container cont) {
		this(ctx, sc, of(cont));
	}

	private static BaseContainerDescriptor of(Container cont) {
		BaseContainerDescriptor desc = BaseContainerDescriptor.create(cont.uid, cont.name, cont.owner, cont.type,
				cont.domainUid, cont.defaultContainer);
		desc.internalId = cont.id;
		return desc;
	}

	@Override
	public void store(final List<AccessControlEntry> entries) {
		try {
			List<AccessControlEntry> compacted = AccessControlEntry.compact(entries);
			List<AccessControlEntry> oldEntries = aclStore.get(container);
			aclStore.store(container, compacted);
			auditLogSupplier.get().ifPresent(auditlog -> {
				ListDifference listDiff = computeListDifference(oldEntries, compacted);
				listDiff.added.forEach(auditlog::logCreate);
				listDiff.removed.forEach(auditlog::logDelete);
			});
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void add(final List<AccessControlEntry> entries) {
		try {
			List<AccessControlEntry> compacted = AccessControlEntry.compact(entries);
			auditLogSupplier.get().ifPresent(auditLog -> compacted.forEach(auditLog::logCreate));
			aclStore.add(container, compacted);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public List<AccessControlEntry> get() {
		try {
			return AccessControlEntry.expand(addOwnerRights(aclStore.get(container)));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void deleteAll() {
		try {
			List<AccessControlEntry> entries = aclStore.get(container);
			auditLogSupplier.get().ifPresent(auditLog -> entries.forEach(auditLog::logDelete));
			aclStore.deleteAll(container);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public List<AccessControlEntry> retrieveAndStore(List<AccessControlEntry> entries) {
		try {
			List<AccessControlEntry> compacted = AccessControlEntry.compact(entries);
			List<AccessControlEntry> oldEntries = aclStore.retrieveAndStore(container, compacted);
			auditLogSupplier.get().ifPresent(auditLog -> {
				ListDifference listDiff = computeListDifference(oldEntries, compacted);
				listDiff.added.forEach(auditLog::logCreate);
				listDiff.removed.forEach(auditLog::logDelete);
			});
			return AccessControlEntry.expand(addOwnerRights(oldEntries));
		} catch (ServerFault e) {
			throw ServerFault.sqlFault(e);

		}
	}

	private List<AccessControlEntry> addOwnerRights(List<AccessControlEntry> list) {
		Set<AccessControlEntry> extended = new HashSet<>(list);
		extended.add(AccessControlEntry.create(container.owner, Verb.All));
		return new ArrayList<>(extended);
	}

	private record ListDifference(List<AccessControlEntry> removed, List<AccessControlEntry> added) {
	}

	private ListDifference computeListDifference(List<AccessControlEntry> oldEntries,
			List<AccessControlEntry> newEntries) {
		List<AccessControlEntry> removed = oldEntries.stream().filter(old -> !newEntries.contains(old)).toList();
		List<AccessControlEntry> added = newEntries.stream().filter(n -> !oldEntries.contains(n)).toList();
		return new ListDifference(removed, added);
	}
}
