/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.service.internal;

import java.io.File;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.GenerationContent;
import net.bluemind.dataprotect.api.GenerationIndex;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.Email;
import net.bluemind.directory.hollow.datamodel.consumer.DirectoryDeserializer;
import net.bluemind.domain.api.Domain;
import net.bluemind.role.api.BasicRoles;

/**
 * Used to read directory entries from "old" PG backup format
 */
public class DirEntriesPgWorker implements IDirEntriesWorker {
	static final Logger logger = LoggerFactory.getLogger(DirEntriesPgWorker.class);

	private final ParGenWorkerRecord partGen;

	public DirEntriesPgWorker(ParGenWorkerRecord partGen) {
		this.partGen = partGen;
	}

	@Override
	public String directoryDataType() {
		return WorkerDataType.DIRECTORY_LEGACY.value;
	}

	@Override
	public String workFilePath(String serverIp) {
		return String.format("/var/backups/bluemind/dp_spool/rsync/%s/bm/core/%d/var/backups/bluemind/work/directory/",
				serverIp, partGen.directory().id);
	}

	@Override
	public void loadContent(String serverIp, GenerationContent geneneration) {
		String path = workFilePath(serverIp);
		List<File> files = loadFiles(path);

		logger.debug("Read files {}", files.stream().map(File::getName).toList());

		for (File snapshot : files) {
			DirectoryDeserializer directoryDeserializer = new DirectoryDeserializer(snapshot, false);

			String domainUid = snapshot.getName();

			if (!partGen.ctx().getSecurityContext().isDomainGlobal()
					&& !domainUid.equals(partGen.ctx().getSecurityContext().getContainerUid())) {
				continue;
			}

			logger.debug("Consuming for domain {}", domainUid);

			Domain domain = new Domain();
			domain.name = domainUid;
			geneneration.domains.add(ItemValue.create(domainUid, domain));

			directoryDeserializer.all().stream().sorted((a, b) -> a.getName().compareTo(b.getName()))
					.filter(isAllowed()).forEach(abRecord -> {
						// ui try to guess the domainUid with the path
						String pathValue = domainUid.concat("/");
						String dataLocation = null;
						if (abRecord.getDataLocation() != null) {
							dataLocation = abRecord.getDataLocation().getServer();
						}
						String email = null;
						if (abRecord.getEmails() != null && !abRecord.getEmails().isEmpty()) {
							email = abRecord.getEmails().stream().filter(Email::getIsDefault).findFirst()
									.orElse(abRecord.getEmails().get(0)).getAddress();
						}
						geneneration.entries.add(new GenerationIndex(abRecord.getUid(), abRecord.getName(), email,
								pathValue, Kind.valueOf(abRecord.getKind()), dataLocation));
					});
		}
	}

	@Override
	public List<File> loadFiles(String path) {
		File dir = new File(path);
		if (!dir.exists()) {
			throw new ServerFault("Unable to load directory listing: '" + path + "' does not exists");
		}
		return Stream.of(dir.listFiles(File::isDirectory)).toList();
	}

	@Override
	public ParGenWorkerRecord getPartGenRecord() {
		return partGen;
	}

	private Predicate<AddressBookRecord> isAllowed() {
		Set<String> allowedUids = new HashSet<>();

		if (!partGen.ctx().getSecurityContext().isDomainGlobal()
				&& !partGen.ctx().getSecurityContext().getRoles().contains(BasicRoles.ROLE_DATAPROTECT)
				&& !partGen.ctx().getSecurityContext().getRoles().contains(BasicRoles.ROLE_MANAGE_RESTORE)) {
			Collection<String> allowedOu = DPService.expandContextManageRestoreOrgUnitPerms(partGen.ctx());

			try {
				allowedUids.addAll(partGen.ctx().getServiceProvider()
						.instance(IDirectory.class, partGen.ctx().getSecurityContext().getContainerUid())
						.search(DirEntryQuery.all()).values.stream().filter(e -> e.value.orgUnitPath != null)
						.filter(e -> e.value.orgUnitPath.path().stream().anyMatch(allowedOu::contains)).map(e -> e.uid)
						.collect(Collectors.toSet()));
			} catch (ServerFault sf) {
				logger.error("Unable to get allowed entries UIDs for {}@{}: {}",
						partGen.ctx().getSecurityContext().getSubject(),
						partGen.ctx().getSecurityContext().getContainerUid(), sf.getMessage(), sf);
			}

			return x -> allowedUids.contains(x.getUid());
		}

		return x -> true;
	}

}
