/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.adm;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import net.bluemind.cli.adm.orphans.report.DomainsReport;
import net.bluemind.cli.adm.orphans.report.Log;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.maintenance.checker.orphan.OrphanChecker;
import net.bluemind.network.topology.Topology;
import picocli.CommandLine.Command;
import picocli.CommandLine.IExitCodeGenerator;
import picocli.CommandLine.Option;

@Command(name = "orphan-check", description = "check and delete orphan mailbox records")
public class OrphanContainerCommand implements ICmdLet, Runnable, IExitCodeGenerator {

	private static final String[] CONTAINER_TYPES = { "mailbox_records" };

	@Option(names = "--workers", description = "run with X workers (default: 4)")
	public int workers = 4;

	@Option(names = "--verbose", description = "show more informations")
	public boolean verbose = false;

	@Option(names = "--auto-repair", description = "Launch missing emails repairs automatically (default: true)")
	public boolean autoRepair = true;

	@Option(names = "--force-delete-many-orphans", description = "Delete orphan emails, even if the total ammount to be deleted is greater than 33%% of the total email data")
	public boolean forceDeleteManyOrphans = false;

	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("maintenance");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return OrphanContainerCommand.class;
		}
	}

	private CliContext ctx;
	protected CliUtils cliUtils;
	protected List<DomainsReport> reports = new ArrayList<>();
	private int exitCode = 0;

	@Override
	public int getExitCode() {
		return exitCode;
	}

	@Override
	public void run() {
		Log.Level logLevel = verbose ? Log.Level.INFO : Log.Level.WARNING;
		ctx.info("Checking orphan containers...");

		DomainsReport report = new DomainsReport("Orphan containers checks", logLevel);
		reports.add(report);

		OrphanChecker orphanChecker = new OrphanChecker(ctx.adminApi(), ctx.infiniteRequestTimeoutAdminApi(),
				Topology.get().any("mail/imap").uid, autoRepair, workers, forceDeleteManyOrphans);
		orphanChecker.check(report, CONTAINER_TYPES, cliUtils.getDomainUids());

		reports.stream().forEach(r -> System.out.println(r.toString())); // NOSONAR: OK
		System.out.flush(); // NOSONAR: OK

		List<String> repairs = reports.stream().map(DomainsReport::repairCommand).filter(repair -> !repair.isEmpty())
				.toList();
		if (!repairs.isEmpty()) {
			ctx.warn("Recommended repair commands to run: ");
			ctx.warn(repairs.stream().distinct().collect(Collectors.joining("\n")));
		}

		List<String> sqlRepairs = reports.stream().map(DomainsReport::sqlCommand).filter(repair -> !repair.isEmpty())
				.toList();
		if (!sqlRepairs.isEmpty()) {
			ctx.warn("Recommended SQL commands to run: ");
			ctx.warn(sqlRepairs.stream().distinct().collect(Collectors.joining("\n")));
		}

	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}
}
