package net.bluemind.exchange.mapi.persistence.pclcache;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.UUID;

import javax.sql.DataSource;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.persistence.IntegerCreator;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.exchange.mapi.api.MapiPCLCacheEntry;
import net.bluemind.exchange.mapi.repository.IMapiPCLCacheStore;

public class MapiPCLCacheStore extends JdbcAbstractStore implements IMapiPCLCacheStore {

	private static final String INSERT = "INSERT INTO t_mapi_pclcache (" + MapiPCLCacheColumns.cols.names()
			+ ") VALUES (" + MapiPCLCacheColumns.cols.values() + ") ON CONFLICT DO NOTHING";
	private static final String SELECT = "SELECT " + MapiPCLCacheColumns.cols.names()
			+ " FROM t_mapi_pclcache WHERE replica_guid=? AND container_item_id=?";
	private static final String EXISTS = "SELECT 1 FROM t_mapi_pclcache WHERE replica_guid=? AND container_item_id=?";
	private static final String DELETE = "DELETE FROM t_mapi_pclcache WHERE replica_guid=? AND container_item_id=?";
	private static final String CLEAN_ORPHANS = """
			WITH orphaned_records AS materialized (
			    SELECT p.ctid
			    FROM t_mapi_pclcache p
			    LEFT JOIN t_container_item c ON p.container_item_id = c.id
			    WHERE c.id IS NULL
			    LIMIT 1000
			)
			DELETE FROM t_mapi_pclcache
			USING orphaned_records
			WHERE t_mapi_pclcache.ctid = orphaned_records.ctid;
			""";

	public MapiPCLCacheStore(DataSource dataSource) {
		super(dataSource);
		logger.debug("Created for ds {}", dataSource);
	}

	@Override
	public void store(UUID replicaGuid, Long globalCounter, MapiPCLCacheEntry entry) throws ServerFault {
		try {
			insert(INSERT, entry, MapiPCLCacheColumns.values(replicaGuid, toContainerItemId(globalCounter)));
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public MapiPCLCacheEntry get(UUID replicaGuid, Long globalCounter) throws ServerFault {
		try {
			return unique(SELECT, rs -> new MapiPCLCacheEntry(), MapiPCLCacheColumns.populator(),
					new Object[] { replicaGuid, toContainerItemId(globalCounter) });
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public boolean contains(UUID replicaGuid, Long globalCounter) throws ServerFault {
		try {
			Integer result = this.unique(EXISTS, IntegerCreator.FIRST, new ArrayList<EntityPopulator<Integer>>(0),
					new Object[] { replicaGuid, toContainerItemId(globalCounter) });
			return result != null && result == 1;
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public void delete(UUID replicaGuid, Long globalCounter) {
		try {
			this.delete(DELETE, new Object[] { replicaGuid, toContainerItemId(globalCounter) });
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	public long cleanOrphan() {
		try {
			return this.delete(CLEAN_ORPHANS, new Object[] {});
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	private long toContainerItemId(long globalCounter) {
		return (globalCounter & 0xff) << 40 | //
				((globalCounter >>> 8 & 0xff) << 32) | //
				((globalCounter >>> 16 & 0xff) << 24) | //
				((globalCounter >>> 24 & 0xff) << 16) | //
				((globalCounter >>> 32 & 0xff) << 8) | //
				(globalCounter >>> 40 & 0xff);
	}
}
