/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.command.settings;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import io.vertx.core.Handler;
import net.bluemind.eas.backend.IBackend;
import net.bluemind.eas.backend.MailboxVacation;
import net.bluemind.eas.dto.IPreviousRequestsKnowledge;
import net.bluemind.eas.dto.OptionalParams;
import net.bluemind.eas.dto.settings.OofMessage;
import net.bluemind.eas.dto.settings.OofMessage.Audience;
import net.bluemind.eas.dto.settings.SettingsRequest;
import net.bluemind.eas.dto.settings.SettingsResponse;
import net.bluemind.eas.dto.settings.SettingsResponse.DeviceInformation;
import net.bluemind.eas.dto.settings.SettingsResponse.Oof;
import net.bluemind.eas.dto.settings.SettingsResponse.UserInformation;
import net.bluemind.eas.dto.settings.SettingsStatus;
import net.bluemind.eas.exception.ServerErrorException;
import net.bluemind.eas.impl.Backends;
import net.bluemind.eas.impl.Responder;
import net.bluemind.eas.protocol.IEasProtocol;
import net.bluemind.eas.serdes.IResponseBuilder;
import net.bluemind.eas.serdes.settings.SettingsRequestParser;
import net.bluemind.eas.serdes.settings.SettingsResponseFormatter;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.eas.wbxml.builder.WbxmlResponseBuilder;

public class SettingsProtocol implements IEasProtocol<SettingsRequest, SettingsResponse> {

	private static final Logger logger = LoggerFactory.getLogger(SettingsProtocol.class);
	private IBackend backend;

	public SettingsProtocol() {
		backend = Backends.dataAccess();
	}

	@Override
	public void parse(BackendSession bs, OptionalParams optParams, Document doc, IPreviousRequestsKnowledge past,
			Handler<SettingsRequest> parserResultHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Parsing *******");
		}

		SettingsRequestParser parser = new SettingsRequestParser();
		SettingsRequest parsed = parser.parse(optParams, doc, past, bs.getLoginAtDomain());
		parserResultHandler.handle(parsed);
	}

	@Override
	public void execute(BackendSession bs, SettingsRequest sr, Handler<SettingsResponse> responseHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Executing *******");
		}

		SettingsResponse response = new SettingsResponse();

		if (sr.userInformation != null && sr.userInformation.get != null) {
			response.status = SettingsStatus.OK;
			UserInformation ui = new SettingsResponse.UserInformation();
			response.userInformation = ui;
			ui.smtpAddresses = bs.getUser().getEmails();
			ui.primaryAddress = bs.getUser().getDefaultEmail();
		}

		if (sr.deviceInformation != null && sr.deviceInformation.set != null) {
			response.status = SettingsStatus.OK;
			DeviceInformation di = new SettingsResponse.DeviceInformation();
			response.deviceInformation = di;
		}

		if (sr.oof != null && sr.oof.get != null) {
			response = builOofGetSettingsResponse(bs, sr);
		}
		if (sr.oof != null && sr.oof.set != null) {
			try {
				backend.getSettingsImporter(bs).setVacation(bs, sr.oof.set);
				response.status = SettingsStatus.OK;
			} catch (ServerErrorException e) {
				logger.error(e.getMessage());
				response.status = SettingsStatus.SERVER_UNAVAILABLE;

			}
		}

		responseHandler.handle(response);
	}

	@Override
	public void write(BackendSession bs, Responder responder, SettingsResponse response,
			final Handler<Void> completion) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Writing *******");
		}

		SettingsResponseFormatter formatter = new SettingsResponseFormatter();
		IResponseBuilder builder = new WbxmlResponseBuilder(bs, responder.asOutput());
		formatter.format(builder, bs.getProtocolVersion(), response, data -> completion.handle(null));
	}

	@Override
	public String address() {
		return "eas.protocol.settings";
	}

	private SettingsResponse builOofGetSettingsResponse(BackendSession bs, SettingsRequest sr) {
		MailboxVacation vacation = backend.getSettingsExporter(bs).getVacation(bs);

		SettingsResponse response = new SettingsResponse();
		response.status = SettingsStatus.OK;

		if (sr.oof.get.bodyType == null) {
			EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
					"The BodyType element is a required child element of the Get element in Settings command Oof\n"
							+ "requests for {}",
					bs.getLoginAtDomain());
			response.status = SettingsStatus.INVALID_ARGUMENTS;
			return response;
		}

		Oof oof = computeOofState(vacation);
		// We don't have currently the choice between several audiences, so the same
		// message will be duplicated
		oof.appliesToInternal = buildOffMessageForBodyType(Oof.BodyType.valueOf(sr.oof.get.bodyType), vacation,
				Audience.AppliesToInternal);
		oof.appliesToExternalKnown = buildOffMessageForBodyType(Oof.BodyType.valueOf(sr.oof.get.bodyType), vacation,
				Audience.AppliesToExternalKnown);
		oof.appliesToExternalUnknown = buildOffMessageForBodyType(Oof.BodyType.valueOf(sr.oof.get.bodyType), vacation,
				Audience.AppliesToExternalUnknown);
		response.oof = oof;
		return response;
	}

	private OofMessage buildOffMessageForBodyType(Oof.BodyType bodyType, MailboxVacation vac, Audience audience) {
		String message = (bodyType.equals(Oof.BodyType.HTML) && vac.textHtml != null) ? vac.textHtml
				: (vac.text != null) ? vac.text : "";
		return new OofMessage(1, message, bodyType, audience);
	}

	private Oof computeOofState(MailboxVacation vac) {
		Oof oof = new SettingsResponse.Oof();
		int oofState = 0;
		if (!vac.enabled) {
			oofState = 0;
		}
		if (vac.enabled && vac.start == null && vac.end == null) {
			oofState = 1;
		}
		if (vac.enabled && vac.start != null && vac.end != null) {
			oofState = 2;
		}
		oof.oofState = oofState;

		// See ASCMD 2.2.3.124
		if (oofState != 2 && vac.start != null && vac.end != null) {
			//
		} else {
			oof.startTime = vac.start;
			oof.endTime = vac.end;
		}
		return oof;
	}

}
