/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.command.search;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import io.vertx.core.Handler;
import net.bluemind.eas.dto.IPreviousRequestsKnowledge;
import net.bluemind.eas.dto.OptionalParams;
import net.bluemind.eas.dto.base.Range;
import net.bluemind.eas.dto.search.SearchRequest;
import net.bluemind.eas.dto.search.SearchResponse;
import net.bluemind.eas.dto.search.SearchResponse.Status;
import net.bluemind.eas.dto.search.SearchResult;
import net.bluemind.eas.dto.search.StoreName;
import net.bluemind.eas.impl.Responder;
import net.bluemind.eas.protocol.IEasProtocol;
import net.bluemind.eas.search.ISearchSource;
import net.bluemind.eas.search.ISearchSource.Results;
import net.bluemind.eas.serdes.IResponseBuilder;
import net.bluemind.eas.serdes.search.SearchRequestParser;
import net.bluemind.eas.serdes.search.SearchResponseFormatter;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.eas.utils.RunnableExtensionLoader;
import net.bluemind.eas.wbxml.builder.WbxmlResponseBuilder;

public class SearchProtocol implements IEasProtocol<SearchRequest, SearchResponse> {

	private static final Logger logger = LoggerFactory.getLogger(SearchProtocol.class);
	private Map<StoreName, Set<ISearchSource>> sources;

	public SearchProtocol() {
		sources = new HashMap<StoreName, Set<ISearchSource>>();
		registerSources();
	}

	@Override
	public void parse(BackendSession bs, OptionalParams optParams, Document doc, IPreviousRequestsKnowledge past,
			Handler<SearchRequest> parserResultHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Parsing *******");
		}
		SearchRequestParser parser = new SearchRequestParser();
		SearchRequest parsed = parser.parse(optParams, doc, past, bs.getLoginAtDomain());
		parserResultHandler.handle(parsed);
	}

	@Override
	public void execute(BackendSession bs, SearchRequest query, Handler<SearchResponse> responseHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Executing *******");
		}

		SearchResponse response = new SearchResponse();
		if (query.store.name == null) {
			EasLogUser.logErrorAsUser(bs.getLoginAtDomain(), logger, "Invalid store name");
			response.status = Status.SERVER_ERROR;
			responseHandler.handle(response);
			return;
		}

		Results<SearchResult> searchResult = search(bs, query);

		response.status = Status.SUCCESS;
		response.store = new SearchResponse.Store();
		response.store.status = SearchResponse.Store.Status.SUCCESS;
		response.store.results = searchResult;
		response.total = searchResult.getNumFound();
		response.range = Range.create(query.store.options.range.min,
				(query.store.options.range.min + response.total.intValue() - 1));

		responseHandler.handle(response);

	}

	private void registerSources() {
		RunnableExtensionLoader<ISearchSource> rel = new RunnableExtensionLoader<>();
		List<ISearchSource> bs = rel.loadExtensions("net.bluemind.eas", "search", "search", "implementation");
		for (ISearchSource ibs : bs) {
			addRegisterSource(ibs.getStoreName(), ibs);
		}
	}

	private void addRegisterSource(StoreName key, ISearchSource value) {
		Set<ISearchSource> set = sources.get(key);
		if (set == null) {
			if (logger.isDebugEnabled()) {
				logger.debug("Add {} in search sources for store {}", value.getClass().getName(), key);
			}
			set = new HashSet<>();
			this.sources.put(key, set);
		}
		set.add(value);
	}

	public Results<SearchResult> search(BackendSession bs, SearchRequest request) {
		Results<SearchResult> ret = new Results<>();
		for (ISearchSource source : sources.get(request.store.name)) {
			Results<SearchResult> rlist = source.search(bs, request);
			ret.setNumFound(rlist.getNumFound());
			ret.addAll(rlist);
		}
		return ret;
	}

	@Override
	public void write(BackendSession bs, Responder responder, SearchResponse response, final Handler<Void> completion) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Writing *******");
		}

		SearchResponseFormatter formatter = new SearchResponseFormatter();
		IResponseBuilder builder = new WbxmlResponseBuilder(bs, responder.asOutput());
		formatter.format(builder, bs.getProtocolVersion(), response, data -> completion.handle(null));
	}

	@Override
	public String address() {
		return "eas.protocol.search";
	}

}
