/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.addressbook.service.internal;

import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import io.vertx.core.buffer.Buffer;
import net.bluemind.addressbook.adapter.AddressbookOwner;
import net.bluemind.addressbook.adapter.ProgressiveVCardBuilder;
import net.bluemind.addressbook.adapter.VCardAdapter;
import net.bluemind.addressbook.adapter.VCardVersion;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCardChanges;
import net.bluemind.addressbook.service.IInCoreIVCardService;
import net.bluemind.core.api.ImportStats;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUpdatesResult;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.core.task.service.LoggingTaskMonitor;
import net.bluemind.core.task.service.NullTaskMonitor;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.TagRef;
import net.fortuna.ical4j.vcard.property.Photo;
import net.fortuna.ical4j.vcard.property.Uid;

public class VCardService implements IInCoreIVCardService {

	private static final Logger logger = LoggerFactory.getLogger(VCardService.class);

	private AddressBookService addressbookService;
	private Container container;
	private BmContext context;

	private RBACManager rbacManager;

	public VCardService(BmContext context, AddressBookService service, Container container) {
		this.context = context;
		this.addressbookService = service;
		this.container = container;
		rbacManager = RBACManager.forContext(context).forContainer(container);

	}

	private String adaptVCards(List<ItemValue<VCard>> cards) {
		StringBuilder sb = new StringBuilder();
		for (ItemValue<VCard> vcard : cards) {
			if (vcard != null) {
				sb.append(adaptCard(vcard).toString());
			}
		}
		return sb.toString();
	}

	@Override
	public String exportAll() throws ServerFault {
		List<ItemValue<VCard>> cards = addressbookService.all();
		return adaptVCards(cards);
	}

	@Override
	public String exportCards(List<String> uids) throws ServerFault {
		// acl checked in addressbookService
		List<ItemValue<VCard>> vcards = addressbookService.multipleGet(uids);
		return adaptVCards(vcards);
	}

	@Override
	public Stream exportAllLight() throws ServerFault {
		List<List<String>> partitioned = Lists.partition(addressbookService.allUids(), 30);
		AtomicInteger index = new AtomicInteger(0);

		GenericStream<String> stream = new GenericStream<String>() {

			@Override
			protected StreamState<String> next() throws Exception {
				if (index.get() == partitioned.size()) {
					return StreamState.end();
				}
				List<String> uids = partitioned.get(index.get());
				String vcf = addressbookService.multipleGetLight(uids).stream().map(VCardService.this::adaptCard)
						.map(Object::toString).reduce("", (s1, s2) -> s1 + s2);
				index.incrementAndGet();
				return StreamState.data(vcf);
			}

			@Override
			protected Buffer serialize(String n) throws Exception {
				return Buffer.buffer(n.getBytes());
			}

		};
		return VertxStream.stream(stream, "text/vcard");

	}

	private net.fortuna.ical4j.vcard.VCard adaptCard(ItemValue<VCard> vcard) {
		net.fortuna.ical4j.vcard.VCard ret = VCardAdapter.adaptCardFromBm(container.uid, vcard.value, VCardVersion.v3);
		try {
			Uid cardUid = new Uid(new LinkedList<>(), container.uid + "," + vcard.uid);
			ret.getProperties().add(cardUid);
		} catch (URISyntaxException | UnsupportedEncodingException e) {
			logger.error(e.getMessage(), e);
		}

		if (vcard.value.identification.photo) {
			try {
				byte[] photo = addressbookService.getPhoto(vcard.uid);
				if (photo != null) {
					ret.getProperties().add(new Photo(photo));
				}
			} catch (ServerFault e) {
				logger.warn(e.getMessage(), e);
			}
		}

		return ret;
	}

	@Override
	public TaskRef importCards(final String vcard) throws ServerFault {
		rbacManager.check(Verb.Write.name());

		return context.provider().instance(ITasksManager.class).run(new BlockingServerTask() {

			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				ImportStats res = importCards(vcard, monitor);
				logger.info("{}/{} vcards imported in {}", res.importedCount(), res.total, container.uid);
			}
		});
	}

	@Override
	public ImportStats directImportCards(String vcard) throws ServerFault {
		rbacManager.check(Verb.Write.name());
		return importCards(vcard, new LoggingTaskMonitor(logger, new NullTaskMonitor(), 0));
	}

	private List<ItemValue<VCard>> readStreamToVCard(String vcard, IServerTaskMonitor monitor) {
		List<net.fortuna.ical4j.vcard.VCard> cards = new ArrayList<>();
		try (ProgressiveVCardBuilder builder = new ProgressiveVCardBuilder(new StringReader(vcard))) {
			monitor.progress(1, "Read vcf file");
			while (builder.hasNext()) {
				try {
					cards.add(builder.next());
				} catch (Exception e) {
					monitor.error("Error occurs trying to import vcard \r\n{}", vcard);
				}
			}
		} catch (Exception e) {
			throw new ServerFault(e);
		}
		List<ItemValue<VCard>> bmCards = new ArrayList<>(cards.size());

		String seed = "" + System.currentTimeMillis();
		for (net.fortuna.ical4j.vcard.VCard card : cards) {
			BaseDirEntry.Kind abOwnerType = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IDirectory.class, container.domainUid).findByEntryUid(container.owner).kind;

			bmCards.add(
					VCardAdapter.adaptCardToBm(card, s -> UUID.nameUUIDFromBytes(seed.concat(s).getBytes()).toString(),
							Optional.of(new AddressbookOwner(container.domainUid, container.owner, abOwnerType)),
							getAllTags()));
		}
		monitor.progress(2, "Parsed " + bmCards.size() + " cards ");
		return bmCards;

	}

	private ImportStats importCards(String vcard, IServerTaskMonitor monitor) throws ServerFault {
		monitor.begin(3, "Begin import");
		List<ItemValue<VCard>> bmCards = readStreamToVCard(vcard, monitor);
		ImportStats ret = doImport(bmCards, monitor.subWork(2));
		monitor.end(true, ret.importedCount() + "/" + ret.total + " vcards imported in " + container.uid,
				JsonUtils.asString(ret));
		return ret;
	}

	private List<TagRef> getAllTags() {
		List<TagRef> allTags = new ArrayList<>();

		BaseDirEntry.Kind abOwnerType = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IDirectory.class, container.domainUid).findByEntryUid(container.owner).kind;

		if (abOwnerType != Kind.ADDRESSBOOK && abOwnerType != Kind.RESOURCE) {
			// owner tags
			allTags.addAll(getTagsService().all().stream()
					.map(tag -> TagRef.create(ITagUids.defaultTags(container.owner), tag))
					.collect(Collectors.toList()));
		}

		// domain tags
		ITags service = context.su().provider().instance(ITags.class, ITagUids.defaultTags(container.domainUid));
		allTags.addAll(service.all().stream().map(tag -> TagRef.create(ITagUids.defaultTags(container.domainUid), tag))
				.collect(Collectors.toList()));

		return allTags;
	}

	private ITags getTagsService() {
		if (container.owner.equals(context.getSecurityContext().getSubject())) {
			return context.getServiceProvider().instance(ITags.class, ITagUids.defaultTags(container.owner));
		} else {
			try (Sudo asUser = new Sudo(container.owner, container.domainUid)) {
				return ServerSideServiceProvider.getProvider(asUser.context).instance(ITags.class,
						ITagUids.defaultTags(container.owner));
			}
		}
	}

	private ImportStats doImport(List<ItemValue<VCard>> bmCards, IServerTaskMonitor monitor) throws ServerFault {

		monitor.begin(bmCards.size(), "Import " + bmCards.size() + " cards");

		ImportStats ret = new ImportStats();
		ret.total = bmCards.size();
		ret.uids = new ArrayList<>(ret.total);

		List<ItemValue<VCard>> toImport = new ArrayList<>(bmCards);

		Iterator<ItemValue<VCard>> it = toImport.iterator();
		VCard card;
		while (it.hasNext()) {
			VCardChanges changes = VCardChanges.create(new ArrayList<>(bmCards.size()), Collections.emptyList(),
					Collections.emptyList());

			while (it.hasNext() && changes.add.size() < 50) {
				ItemValue<VCard> next = it.next();
				card = next.value;
				String uid = next.uid;
				changes.add.add(VCardChanges.ItemAdd.create(uid, card));

			}
			ContainerUpdatesResult resp = addressbookService.updates(changes);
			ret.uids.addAll(resp.added);

			monitor.progress(changes.add.size(), "in progress");
		}
		return ret;
	}

}
