/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental.persistence;

import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;

public class IncrementalIndexingStore extends JdbcAbstractStore implements IIncrementalIndexingStore {

	private static final Logger logger = LoggerFactory.getLogger(IncrementalIndexingStore.class);

	public IncrementalIndexingStore(BmContext ctx) {
		super(ctx.getDataSource());
	}

	/**
	 * returns null if the container or its location is unknown
	 */
	@Override
	public ContainerSyncState getState(ContainerUid cuid) throws SQLException {
		Long version = unique("select version from t_indexed_versions where container_uid=?", r -> r.getLong(1),
				Collections.emptyList(), new Object[] { cuid.value() });
		if (version != null) {
			return new ContainerSyncState(cuid, version.longValue());
		} else {
			return new ContainerSyncState(cuid, 0L);
		}
	}

	@Override
	public void markDirty(ContainerUid c) throws SQLException {
		insert("""
				insert into t_dirty_indices (container_uid)
				values (?)
				on conflict (container_uid)
				do update set dirtied_at=now()
				""", new Object[] { c.value() });
	}

	@Override
	public boolean checkpointSync(ContainerSyncState stateUpdate, DirtynessMarker dirt) throws SQLException {
		return ServerFault.onException(() -> {
			update("""
					insert into t_indexed_versions (container_uid, version) values (?, ?)
					on conflict (container_uid)
					do update set version = ?
					""", new Object[] { stateUpdate.cont().value(), stateUpdate.version(), stateUpdate.version() });
			return delete("delete from t_dirty_indices where container_uid = ? and dirtied_at <= ?",
					new Object[] { dirt.container().value(), dirt.since() }) > 0;
		}, ErrorCode.SQL_ERROR);
	}

	@Override
	public List<DirtynessMarker> fetchNexts(Set<String> excludeIds, int limit) throws SQLException {
		return select(
				"SELECT container_uid, dirtied_at FROM t_dirty_indices WHERE container_uid != ALL(?) ORDER BY dirtied_at ASC LIMIT ?",
				r -> new DirtynessMarker(ContainerUid.of(r.getString(1)), r.getTimestamp(2).toInstant()),
				Collections.emptyList(), new Object[] {
						Optional.ofNullable(excludeIds).orElse(Collections.emptySet()).toArray(String[]::new), limit });
	}

	@Override
	public void forgetStates() {
		TxEnabler.atomically(() -> ServerFault.onExceptionVoid(() -> {
			update("truncate t_indexed_versions", new Object[] {});
			update("truncate t_dirty_indices", new Object[] {});
		}, ErrorCode.SQL_ERROR));
		logger.info("Indexing states forgotten");
	}

	@Override
	@VisibleForTesting
	public long slowCostlyCount() throws SQLException {
		return unique("select count(*) from t_dirty_indices", r -> r.getLong(1), Collections.emptyList(),
				new Object[] {});
	}

}
