/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.notes.impl;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.notes.impl.pg.RestoreNotesTaskPg;
import net.bluemind.dataprotect.service.BackupPath;
import net.bluemind.network.topology.Topology;
import net.bluemind.notes.api.INote;
import net.bluemind.notes.api.INoteUids;
import net.bluemind.notes.api.VNote;
import net.bluemind.notes.api.VNoteChanges;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class RestoreNotesTask extends BlockingServerTask implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreNotesTask.class);

	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;

	public RestoreNotesTask(DataProtectGeneration backup, Restorable item) {
		this.backup = backup;
		this.restorableItem = new RestoreRestorableItem(item);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.NOTES.name()));

		NotesBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreNotesTaskPg(backup, restorableItem).run();
		}
	}

	protected NotesBackupRepository getRepository() {
		try {
			List<PartGeneration> parts = backup.parts;
			PartGeneration corepart = parts.stream().filter(p -> WorkerDataType.NOTES == p.getWorkerDataType())
					.findFirst().orElseThrow(() -> ServerFault.notFound("Unable to find backup part 'notes'"));
			ItemValue<Server> coreServer = Topology.get().core();
			Path backupPath = Paths.get(BackupPath.get(coreServer, TagDescriptor.bm_core.getTag()),
					String.valueOf(corepart.id), "var/backups/bluemind/notes");
			if (Files.exists(backupPath)) {
				return new NotesBackupRepository(backupPath);
			}
		} catch (Exception e) {
			logger.info(
					"Unable to get notes backup repository: {}, this message can be ignored if the restore process is related to a legacy database backup",
					e.getMessage());
		}
		return null;
	}

	private void restore(NotesBackupRepository repository) {
		BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
		IContainers containerApi = live.provider().instance(IContainers.class);

		try {
			List<RestorableNotes> allNotes = repository.getRestorableNotes(restorableItem.domain(),
					restorableItem.entryUid());
			restorableItem.monitorLog("Restoring {} notes", allNotes.size());
			List<ContainerDescriptor> liveLists = containerApi
					.all(ContainerQuery.ownerAndType(restorableItem.liveEntryUid(), INoteUids.TYPE));

			for (RestorableNotes rn : allNotes) {
				if (liveLists.stream().anyMatch(c -> c.uid.equals(rn.getContainer().uid))) {
					INote noteApi = live.provider().instance(INote.class, rn.getContainer().uid);
					noteApi.reset();
					restorableItem.monitorLog("notes reset done");
				} else {
					containerApi.create(rn.getContainer().uid, rn.getContainer());
					restorableItem.monitorLog("notes recreated");
				}

				INote noteApi = live.provider().instance(INote.class, rn.getContainer().uid);
				for (List<ItemValue<VNote>> batch : Lists.partition(rn.getItems(), 1000)) {
					VNoteChanges changes = VNoteChanges.create(batch.stream()
							.map(e -> VNoteChanges.ItemAdd.create(e.uid, e.value, false)).collect(Collectors.toList()),
							Collections.emptyList(), Collections.emptyList());
					noteApi.updates(changes);
					restorableItem.monitorLog("{} notes to restore ", batch.size());
				}
			}
		} catch (Exception e) {
			logger.error("Error while restoring notes", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.NOTES));
		} finally {
			restorableItem.endTask();
		}
	}
}
