/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.user.service.internal;

import java.util.Locale;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.system.api.IGlobalSettings;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;

public abstract class BaseUserService implements IUser {

	protected final BmContext bmContext;
	protected final String domainName;

	private static final Logger logger = LoggerFactory.getLogger(BaseUserService.class);

	protected BaseUserService(BmContext ctx, ItemValue<Domain> domain) {
		this.bmContext = ctx;
		this.domainName = domain.uid;
	}

	@Override
	public final String getLocale(String uid) throws ServerFault {
		IServiceProvider provider = bmContext.su().provider();

		try {
			IUserSettings userSettings = provider.instance(IUserSettings.class, domainName);
			return Optional.ofNullable(userSettings.getOne(uid, "lang")).orElseGet(() -> {
				IGlobalSettings settingsGlobal = provider.instance(IGlobalSettings.class);
				Optional<String> globalLang = Optional.ofNullable(settingsGlobal.get().get("lang"));
				return globalLang.orElse(Locale.ENGLISH.getLanguage());
			});
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return Locale.ENGLISH.getLanguage();
		}
	}

}
