package net.bluemind.mailbox.service.testhelper;

import java.io.IOException;
import java.io.InputStream;

import com.google.common.io.ByteStreams;

import io.vertx.core.buffer.Buffer;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.IMailboxItems;
import net.bluemind.backend.mail.api.ImapAck;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxItem;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Part;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.core.container.api.IOfflineMgmt;
import net.bluemind.core.container.api.IdRange;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;

public class MailItemTestHelper {

	public record CreateMailResult(long id, MailboxItem mailBoxItem, ItemValue<MailboxFolder> folder,
			ItemValue<Mailbox> mailbox) {
	}

	public static CreateMailResult createMail(InputStream mailInputStream, String domain, String userId,
			String folderName) throws IOException {
		ServerSideServiceProvider serviceProvider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		// Mailbox
		IMailboxes mailboxApi = serviceProvider.instance(IMailboxes.class, domain);
		ItemValue<Mailbox> mailbox = mailboxApi.getComplete(userId);

		// Get INBOX folder
		CyrusPartition part = CyrusPartition.forServerAndDomain(mailbox.value.dataLocation, domain);
		IDbReplicatedMailboxes replicateMailboxesApi = serviceProvider.instance(IDbReplicatedMailboxes.class, part.name,
				"user." + mailbox.uid.replace('.', '^'));
		ItemValue<MailboxFolder> folder = replicateMailboxesApi.byName(folderName);

		CreateMailResult result = createMail(mailInputStream, folder, domain, userId);
		return new CreateMailResult(result.id, result.mailBoxItem, folder, mailbox);
	}

	public static CreateMailResult createMail(InputStream mailInputStream, ItemValue<MailboxFolder> folder,
			String domain, String userId) throws IOException {
		ServerSideServiceProvider serviceProvider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		// Create part and mailbox record
		IMailboxItems mailboxItemService = serviceProvider.instance(IMailboxItems.class, folder.uid);
		net.bluemind.core.api.Stream stream = VertxStream
				.stream(Buffer.buffer(ByteStreams.toByteArray(mailInputStream)));
		String partId = mailboxItemService.uploadPart(stream);
		Part fullEml = Part.create(null, "message/rfc822", partId);
		MailboxItem mailboxRecord = new MailboxItem();
		mailboxRecord.body = MessageBody.of("subject", fullEml);

		// save the mailbox record
		IOfflineMgmt offlineMgmt = serviceProvider.instance(IOfflineMgmt.class, domain, userId);
		IdRange oneId = offlineMgmt.allocateOfflineIds(1);
		long expectedId = oneId.globalCounter;
		ImapAck result = mailboxItemService.createById(expectedId, mailboxRecord);

		mailboxRecord.imapUid = result.imapUid;
		return new CreateMailResult(expectedId, mailboxRecord, folder, null);
	}

	public static void deleteMail(long mailboxItemId, String domain, String userId, String folderName) {
		ServerSideServiceProvider serviceProvider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IMailboxes mailboxApi = serviceProvider.instance(IMailboxes.class, domain);
		ItemValue<Mailbox> mailbox = mailboxApi.getComplete(userId);
		CyrusPartition part = CyrusPartition.forServerAndDomain(mailbox.value.dataLocation, domain);
		IDbReplicatedMailboxes replicateMailboxesApi = serviceProvider.instance(IDbReplicatedMailboxes.class, part.name,
				"user." + mailbox.uid.replace('.', '^'));
		ItemValue<MailboxFolder> folder = replicateMailboxesApi.byName(folderName);
		IMailboxItems mailboxItemService = serviceProvider.instance(IMailboxItems.class, folder.uid);
		mailboxItemService.deleteById(mailboxItemId);
	}
}
