/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.central.reverse.proxy.vertx.impl;

import static net.bluemind.central.reverse.proxy.common.ProxyEventBusAddress.ADDRESS;
import static net.bluemind.central.reverse.proxy.common.ProxyEventBusAddress.MODEL_READY_NAME;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.ACCEPT_BACKLOG;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.HTTP2_MAX_STREAMS;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.KEEP_ALIVE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.MAX_POOL_SIZE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.MAX_WEB_SOCKETS;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.PORT;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.SERVER_PROXY_PROTOCOL;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.SERVER_USE_ALPN;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.TCP_FASTOPEN;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.TCP_KEEP_ALIVE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.TCP_NO_DELAY;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.TCP_QUICKACK;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.WEBSOCKETPORT;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.ACTIVE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.ENGINE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.TRUST_ALL;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.USE_ALPN;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.VERIFY_HOST;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Proxy.Ssl.Engine.OPEN_SSL;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Promise;
import io.vertx.core.eventbus.MessageConsumer;
import io.vertx.core.http.Http2Settings;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpServer;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.json.JsonObject;
import io.vertx.core.net.JdkSSLEngineOptions;
import io.vertx.core.net.OpenSSLEngineOptions;
import io.vertx.core.net.SSLEngineOptions;
import io.vertx.httpproxy.HttpProxy;
import net.bluemind.central.reverse.proxy.model.client.ProxyInfoStoreClient;

public class ProxyVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(ProxyVerticle.class);
	static AtomicReference<String> hostname = new AtomicReference<>();

	private final Config config;
	private MessageConsumer<JsonObject> vertxConsumer;

	public ProxyVerticle(Config config) {
		this.config = config;
	}

	@Override
	public void start(Promise<Void> p) {
		logger.info("[proxy:{}] Starting", deploymentID());
		vertxConsumer = vertx.eventBus().<JsonObject>consumer(ADDRESS).handler(event -> {
			if (MODEL_READY_NAME.equals(event.headers().get("action"))) {
				logger.info("[proxy:{}] Model ready, starting verticle instance proxy", deploymentID());
				startProxy();
				logger.info("[proxy:{}] Started on port {} and {}", deploymentID(), config.getInt(PORT),
						config.getInt(WEBSOCKETPORT));
			}
		});
		p.complete();
	}

	private int getPoolSize() {
		int poolsize = config.getInt(MAX_POOL_SIZE);
		int websockets = config.getInt(MAX_WEB_SOCKETS);
		if (websockets > poolsize) {
			int newpoolsize = websockets + 1000;
			logger.error(
					"CRP configuration has {} maximum websocket but {} maximum pool size. Using {} as pool size instead.",
					websockets, poolsize, newpoolsize);
			return newpoolsize;
		}
		return poolsize;
	}

	private static String getHostname() {
		String localHostname = hostname.get();
		if (localHostname == null) {
			try {
				localHostname = InetAddress.getLocalHost().getCanonicalHostName();
			} catch (UnknownHostException e) {
				logger.error("Unable to determine current CRP Hostname : {}", e);
				localHostname = "Unknown";
			}
			hostname.set(localHostname);
			logger.info("Current CRP Hostname (X-BM-CRP) : {}", localHostname);
		}
		return localHostname;
	}

	private void startProxy() {
		/*
		 * We have two different servers because websocket does not play nice with
		 * HTTP2, we need to set specific settings for the client/server options
		 * 
		 * By default, listens on 8080 for normal requests, and 8079 for websocket
		 * requests, but this NOT enforced
		 */
		startHttpProxy();
		startWebsocketProxy();
	}

	private void startHttpProxy() {
		HttpClient proxyClient = vertx.createHttpClient(httpClientOptions());
		HttpProxy httpProxy = HttpProxy.reverseProxy(proxyClient);
		ProxyInfoStoreClient storeClient = ProxyInfoStoreClient.create(vertx);
		httpProxy.originSelector(new DownstreamSelector(storeClient));
		httpProxy.addInterceptor(new CRPInterceptor(getHostname()));
		Http2Settings http2Settings = new Http2Settings().setMaxConcurrentStreams(config.getInt(HTTP2_MAX_STREAMS));
		HttpServerOptions serverOptions = new HttpServerOptions() //
				.setAcceptBacklog(config.getInt(ACCEPT_BACKLOG)) //
				.setTcpKeepAlive(config.getBoolean(TCP_KEEP_ALIVE)) //
				.setTcpNoDelay(config.getBoolean(TCP_NO_DELAY)) //
				.setTcpFastOpen(config.getBoolean(TCP_FASTOPEN)) //
				.setTcpQuickAck(config.getBoolean(TCP_QUICKACK)) //
				.setUseProxyProtocol(config.getBoolean(SERVER_PROXY_PROTOCOL)) //
				.setInitialSettings(http2Settings) //
				.setAlpnVersions(List.of(HttpVersion.HTTP_2, HttpVersion.HTTP_1_1)) //
				.setUseAlpn(config.getBoolean(SERVER_USE_ALPN)) //
				.setHttp2ClearTextEnabled(true) //
				.setReuseAddress(true); //
		HttpServer proxyServer = vertx.createHttpServer(serverOptions);
		proxyServer.requestHandler(httpProxy).listen(config.getInt(PORT));
	}

	private void startWebsocketProxy() {
		HttpClient proxyClient = vertx.createHttpClient(websocketHttpClientOptions());
		HttpProxy httpProxy = HttpProxy.reverseProxy(proxyClient);
		ProxyInfoStoreClient storeClient = ProxyInfoStoreClient.create(vertx);
		httpProxy.originSelector(new DownstreamSelector(storeClient));
		httpProxy.addInterceptor(new CRPInterceptor(getHostname()));
		HttpServerOptions websocketServerOptions = new HttpServerOptions() //
				.setAcceptBacklog(config.getInt(ACCEPT_BACKLOG)) //
				.setTcpKeepAlive(config.getBoolean(TCP_KEEP_ALIVE)) //
				.setTcpNoDelay(config.getBoolean(TCP_NO_DELAY)) //
				.setTcpFastOpen(config.getBoolean(TCP_FASTOPEN)) //
				.setTcpQuickAck(config.getBoolean(TCP_QUICKACK)) //
				.setUseProxyProtocol(config.getBoolean(SERVER_PROXY_PROTOCOL)) //
				.setUseAlpn(false) //
				.setReuseAddress(true); //
		HttpServer websocketProxyServer = vertx.createHttpServer(websocketServerOptions);
		websocketProxyServer.requestHandler(httpProxy).listen(config.getInt(WEBSOCKETPORT));
	}

	private HttpClientOptions httpClientOptions() {
		SSLEngineOptions sslEngineOptions = (config.getString(ENGINE).toUpperCase().equals(OPEN_SSL.name()))
				? new OpenSSLEngineOptions()
				: new JdkSSLEngineOptions();
		return new HttpClientOptions() //
				.setKeepAlive(config.getBoolean(KEEP_ALIVE)) //
				.setTcpKeepAlive(config.getBoolean(TCP_KEEP_ALIVE)) //
				.setTcpNoDelay(config.getBoolean(TCP_NO_DELAY)) //
				.setMaxPoolSize(getPoolSize()) //
				.setMaxWebSockets(config.getInt(MAX_WEB_SOCKETS)) //
				.setHttp2ClearTextUpgrade(true) //
				.setHttp2MaxPoolSize(config.getInt(MAX_POOL_SIZE)) //
				.setProtocolVersion(HttpVersion.HTTP_1_1) //
				.setReusePort(true) //
				.setTcpFastOpen(config.getBoolean(TCP_FASTOPEN)) //
				.setSsl(config.getBoolean(ACTIVE)) //
				.setUseAlpn(config.getBoolean(USE_ALPN)) //
				.setSslEngineOptions(sslEngineOptions) //
				.setTrustAll(config.getBoolean(TRUST_ALL)) //
				.setVerifyHost(config.getBoolean(VERIFY_HOST));
	}

	private HttpClientOptions websocketHttpClientOptions() {
		SSLEngineOptions sslEngineOptions = (config.getString(ENGINE).toUpperCase().equals(OPEN_SSL.name()))
				? new OpenSSLEngineOptions()
				: new JdkSSLEngineOptions();
		return new HttpClientOptions() //
				.setKeepAlive(config.getBoolean(KEEP_ALIVE)) //
				.setTcpKeepAlive(config.getBoolean(TCP_KEEP_ALIVE)) //
				.setTcpNoDelay(config.getBoolean(TCP_NO_DELAY)) //
				.setMaxPoolSize(getPoolSize()) //
				.setMaxWebSockets(config.getInt(MAX_WEB_SOCKETS)) //
				.setProtocolVersion(HttpVersion.HTTP_1_1) //
				.setReusePort(true) //
				.setPipelining(false) //
				.setTcpFastOpen(config.getBoolean(TCP_FASTOPEN)) //
				.setSsl(config.getBoolean(ACTIVE)) //
				.setUseAlpn(false) //
				.setSslEngineOptions(sslEngineOptions) //
				.setTrustAll(config.getBoolean(TRUST_ALL)) //
				.setVerifyHost(config.getBoolean(VERIFY_HOST));
	}

	public void tearDown() {
		if (vertxConsumer != null) {
			vertxConsumer.unregister();
		}
	}
}
