/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.central.reverse.proxy.vertx.impl;

import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.DefaultCookie;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import io.netty.handler.codec.http.cookie.ServerCookieEncoder;
import io.vertx.core.Future;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.httpproxy.Body;
import io.vertx.httpproxy.ProxyContext;
import io.vertx.httpproxy.ProxyInterceptor;
import io.vertx.httpproxy.ProxyRequest;
import io.vertx.httpproxy.ProxyResponse;
import net.bluemind.central.reverse.proxy.vertx.Auth;

public class CRPInterceptor implements ProxyInterceptor {
	private static final Logger requestsLogger = LoggerFactory.getLogger("net.bluemind.central.reverse.proxy.requests");
	public static final String COOKIENAME = "BMCRP";

	private static final Logger logger = LoggerFactory.getLogger(CRPInterceptor.class);
	private String hostname;

	public CRPInterceptor(String hostname) {
		this.hostname = hostname;
	}

	@Override
	public Future<ProxyResponse> handleProxyRequest(ProxyContext context) {
		ProxyRequest request = context.request();
		String auth = context.request().headers().get(HttpHeaders.AUTHORIZATION);
		if (auth != null && !auth.isEmpty() && auth.startsWith("Basic")) {
			return loginFromBasicAuth(context, auth);
		}
		URI uri = URI.create(request.getURI());
		String requestPath = uri.getPath();
		if (requestPath.startsWith("/keycloak/realms") && requestPath.endsWith("/login-actions/authenticate")
				&& request.getMethod().equals(HttpMethod.POST)
				&& "application/x-www-form-urlencoded".equalsIgnoreCase(request.headers().get("Content-Type"))) {
			return loginFromKeycloak(context);
		}
		return loginFromCookie(context);
	}

	/* Login MAPI */
	private Future<ProxyResponse> loginFromBasicAuth(ProxyContext context, String auth) {
		byte[] credDecoded = Base64.getDecoder().decode(auth.substring("Basic".length()).trim());
		String credentials = new String(credDecoded, StandardCharsets.UTF_8);
		String[] loginPassword = credentials.split(":", 2);
		context.set("auth", new Auth(loginPassword[0]));
		return context.sendRequest();
	}

	/* Login all web applications */
	private Future<ProxyResponse> loginFromCookie(ProxyContext context) {
		Optional.ofNullable(context.request().headers().get(HttpHeaders.COOKIE))
				.ifPresent(cookies -> ServerCookieDecoder.LAX.decode(cookies).stream() //
						.filter(cookie -> COOKIENAME.equals(cookie.name())) //
						.findFirst().map(Cookie::value) //
						.map(Auth::new) //
						.ifPresent(auth -> context.set("auth", auth)));
		return context.sendRequest();
	}

	/* Detect username login all web applications, on keycloak URIs */
	private Future<ProxyResponse> loginFromKeycloak(ProxyContext context) {
		Body body = context.request().getBody();
		BufferingWriteStream bws = new BufferingWriteStream();
		return body.stream().pipeTo(bws).compose(r -> {
			Buffer bodyContent = bws.content();
			String bodyString = bodyContent.toString(StandardCharsets.UTF_8);
			Map<String, String> formParams = decodeFormData(bodyString);
			Optional.ofNullable(formParams.get("username")).map(Auth::new).ifPresent(auth -> {
				context.set("auth", auth);
				Cookie cookie = new DefaultCookie(COOKIENAME, auth.login());
				cookie.setPath("/");
				cookie.setSecure(true);
				context.set("addcookie", ServerCookieEncoder.LAX.encode(cookie));
			});
			context.request().setBody(Body.body(bodyContent));
			return context.sendRequest();
		});
	}

	@Override
	public Future<Void> handleProxyResponse(ProxyContext context) {
		Optional.ofNullable(context.get("addcookie", String.class))
				.ifPresent(cookie -> context.response().headers().add(HttpHeaders.SET_COOKIE, cookie));
		Optional<Auth> maybeAuth = Optional.ofNullable(context.get("auth", Auth.class));

		context.response().putHeader("X-BM-CRP", hostname);

		return context.sendResponse().andThen(ar -> {
			HttpServerRequest proxiedRequest = context.request().proxiedRequest();
			if (ar.succeeded()) {
				if (requestsLogger.isInfoEnabled()) {
					requestsLogger.info("[{}] {} {} - {} {} => {} {}", maybeAuth.map(Auth::login).orElse("anonymous"),
							proxiedRequest.version(), //
							proxiedRequest.connection().remoteAddress(), //
							proxiedRequest.method(), //
							proxiedRequest.path(), //
							context.response().getStatusCode(), //
							context.response().getStatusMessage());
				}
			} else {
				requestsLogger.error("[{}] {} {} - {} {} => {} FAILED: {}",
						maybeAuth.map(Auth::login).orElse("anonymous"), //
						proxiedRequest.version(), //
						proxiedRequest.connection().remoteAddress(), //
						proxiedRequest.method(), //
						proxiedRequest.path(), //
						context.response().getStatusCode(), //
						context.response().getStatusMessage(), //
						ar.cause());
			}
		});
	}

	private Map<String, String> decodeFormData(String formData) {
		Map<String, String> params = new HashMap<>();
		try {
			String[] pairs = formData.split("&");
			for (String pair : pairs) {
				String[] keyValue = pair.split("=", 2);
				String key = URLDecoder.decode(keyValue[0], StandardCharsets.UTF_8.name());
				String value = keyValue.length > 1 ? URLDecoder.decode(keyValue[1], StandardCharsets.UTF_8.name()) : "";
				params.put(key, value);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return params;
	}
}