package net.bluemind.directory.hollow.datamodel.producer.multicore.keydb;

import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.adressbookRecordHKey;
import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.directoryKey;
import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.offlineDirectoryKey;
import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.versionKey;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.lettuce.core.RedisFuture;
import io.lettuce.core.api.async.RedisAsyncCommands;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.producer.multicore.IMultiCoreDirectorySerializerStore;

public class KeyDbMultiCoreDirectorySerializerStore implements IMultiCoreDirectorySerializerStore {

	private Logger logger = LoggerFactory.getLogger(KeyDbMultiCoreDirectorySerializerStore.class);
	private String domainUid;
	private RedisConnections connections;

	public KeyDbMultiCoreDirectorySerializerStore(String domainUid) {
		this.connections = RedisConnections.factory.get();
		this.domainUid = domainUid;
	}

	@Override
	public long getVersion() {
		return Optional.ofNullable(connections.getVersionCommand().get(versionKey(domainUid))).map(Long::parseLong)
				.orElse(0l);
	}

	@Override
	public void finishSerialization(long newVersion) {
		logger.info("Finish serialization, set new version {} for {}", newVersion, domainUid);
		connections.getVersionCommand().set(versionKey(domainUid), String.valueOf(newVersion));
	}

	@Override
	public void batchSaveDeleteAdressBookRecord(long version, List<AddressBookRecord> createRecords,
			List<String> deletedUids) {
		String key = directoryKey(domainUid, version);

		RedisAsyncCommands<String, AddressBookRecord> cmd = connections.getAsyncRecordCommand();

		RedisFuture<?> lastFuture = null;
		try {
			for (AddressBookRecord createRecord : Optional.ofNullable(createRecords).orElse(List.of())) {
				lastFuture = cmd.hset(key, adressbookRecordHKey(createRecord.uid), createRecord);
			}

			for (String deletedUid : Optional.ofNullable(deletedUids).orElse(List.of())) {
				lastFuture = cmd.hdel(key, adressbookRecordHKey(deletedUid));
			}

			if (lastFuture != null) {
				lastFuture.get(5, TimeUnit.SECONDS);
			}
		} catch (Exception e) {
			throw new ServerFault("Failed to batch save/delete address book records", e);
		}
	}

	@Override
	public AddressBookRecord getAdressBookRecord(long version, String uid) {
		return connections.getRecordCommand().hget(directoryKey(domainUid, version), adressbookRecordHKey(uid));
	}

	@Override
	public void copyAdressBookRecords(long version, long newVersion) {
		connections.getRecordCommand().copy(directoryKey(domainUid, version), directoryKey(domainUid, newVersion));
	}

	@Override
	public void deleteAllAdressBookRecords(long oldVersion) {
		logger.info("Delete all address books for version {} on domain {}", oldVersion, domainUid);
		connections.getRecordCommand().del(directoryKey(domainUid, oldVersion));
		connections.getOfflineRecordCommand().del(offlineDirectoryKey(domainUid, oldVersion));
		connections.getVersionCommand().del(versionKey(domainUid));
	}

	@Override
	public void saveOfflineAddressBook(long version, OfflineAddressBook oab) {
		connections.getOfflineRecordCommand().set(offlineDirectoryKey(domainUid, version), oab);
	}

	@Override
	public void setAdressBookRecordExpire(long version, Optional<Duration> ttl) {
		if (ttl.isPresent()) {
			connections.getRecordCommand().expire(directoryKey(domainUid, version), ttl.get());
			connections.getOfflineRecordCommand().expire(offlineDirectoryKey(domainUid, version), ttl.get());
		} else {
			connections.getRecordCommand().persist(directoryKey(domainUid, version));
			connections.getOfflineRecordCommand().persist(offlineDirectoryKey(domainUid, version));
		}
	}
}
