package net.bluemind.directory.hollow.datamodel.producer.multicore;

import java.io.File;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.hollow.api.consumer.HollowConsumer.BlobRetriever;

import net.bluemind.config.InstallationId;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.DataLocation;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.producer.EntryToAdressBookMapper;
import net.bluemind.directory.hollow.datamodel.producer.IDirectorySerializer;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;

public class MultiCoreDirectorySerializer implements IDirectorySerializer {

	private static final int BATCH_SIZE = 100;
	private static final boolean DROP_HIDDEN = new File("/etc/bm/hollow.no.hidden").exists();
	private static final Logger logger = LoggerFactory.getLogger(MultiCoreDirectorySerializer.class);

	private ServerSideServiceProvider prov;

	private record Locks(Object produceLock, Object rebuildLock) {
	}

	private static final Map<String, Locks> lockByDomain = new ConcurrentHashMap<>();
	private final IMultiCoreDirectorySerializerStore store;
	private final String domainUid;

	private long throttleMs;

	public MultiCoreDirectorySerializer(String domainUid, IMultiCoreDirectorySerializerStore directoryStore,
			long throttleMs) {
		this.store = directoryStore;
		this.domainUid = domainUid;
		this.prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		this.throttleMs = throttleMs;
	}

	public void init() {
		// Only DirectorySerializer need code there
	}

	@Override
	public void start() {
		produce();
	}

	@Override
	public void remove() {
		synchronized (getLock(domainUid).produceLock()) {
			long currentVersion = store.getVersion();
			store.deleteAllAdressBookRecords(currentVersion);
		}
	}

	@Override
	public long produce() {
		synchronized (getLock(domainUid).produceLock()) {
			return serializeIncrement();
		}
	}

	@Override
	public BlobRetriever getBlobRetriever() {
		return null;
	}

	@Override
	public long getLastVersion() {
		return store.getVersion();
	}

	private static Locks getLock(String domainUid) {
		return lockByDomain.computeIfAbsent(domainUid, key -> new Locks(new Object(), new Object()));
	}

	private long serializeIncrement() {
		long currentVersion = store.getVersion();
		logger.info("Start serialization for version {} for domain {}", currentVersion, domainUid);

		IDirectory dirApi = prov.instance(IDirectory.class, domainUid);
		ContainerChangeset<String> changeset = dirApi.changeset(currentVersion);
		long nextVersion = changeset.version;
		if (currentVersion != nextVersion) {
			try {
				IMailboxes mboxApi = prov.instance(IMailboxes.class, domainUid);

				changeset.created = new ArrayList<>(changeset.created);
				changeset.created.addAll(changeset.updated);
				List<String> allUids = new ArrayList<>(
						Sets.newHashSet(Iterables.concat(changeset.created, changeset.updated)));
				logger.info("Sync from v{} gave +{} / -{} uid(s)", currentVersion, allUids.size(),
						changeset.deleted.size());

				ItemValue<Domain> domain = prov.instance(IInCoreDomains.class).getUnfiltered(domainUid);

				// copy old version to new version
				store.copyAdressBookRecords(currentVersion, nextVersion);

				OfflineAddressBook oab = EntryToAdressBookMapper.createOabEntry(domain, changeset.version);
				store.saveOfflineAddressBook(nextVersion, oab);

				// apply update/create changeset
				Map<String, DataLocation> locationCache = new ConcurrentHashMap<>();
				List<CompletableFuture<Void>> futures = new ArrayList<>(allUids.size());
				ExecutorService pool = Executors.newVirtualThreadPerTaskExecutor();
				for (String uid : allUids) {
					futures.add(CompletableFuture
							.runAsync(() -> loadOne(dirApi, nextVersion, mboxApi, domain, locationCache, uid), pool));
				}
				CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).orTimeout(1, TimeUnit.HOURS).join();

				// apply delete changeset
				for (List<String> deletedUids : Lists.partition(changeset.deleted, BATCH_SIZE)) {
					store.batchSaveDeleteAdressBookRecord(nextVersion, List.of(), deletedUids);
				}
				store.finishSerialization(nextVersion);
				try {
					store.setAdressBookRecordExpire(currentVersion, Optional.of(Duration.ofMillis(throttleMs * 2l)));
				} catch (Exception e) {
					logger.error("Not able to clean old version of addressbook", e);
				}
			} catch (Exception e) {
				try {
					store.deleteAllAdressBookRecords(nextVersion);
				} catch (Exception deleteException) {
					logger.error("Not able to clean new addressbook");
				}
				throw e;
			}
		}
		logger.info("Serialization done, switch to version {} for domain {}", nextVersion, domainUid);
		return nextVersion;
	}

	private void loadOne(IDirectory dirApi, long nextVersion, IMailboxes mboxApi, ItemValue<Domain> domain,
			final Map<String, DataLocation> locationCache, String uid) {
		ItemValue<DirEntry> entry = dirApi.findItemValueByEntryUid(uid);
		if (entry == null || entry.value == null || !supportedType(entry)) {
			return;
		}

		ItemValue<Mailbox> mailbox = mboxApi.getComplete(uid);
		List<String> deletedUids = Collections.emptyList();
		List<ItemValue<DirEntry>> createUpdateDirEntry = Collections.emptyList();

		if (dropHiddenEntry(entry)) {
			deletedUids = Collections.singletonList(uid);
		} else {
			createUpdateDirEntry = Collections.singletonList(entry);
		}

		List<AddressBookRecord> createUpdate = EntryToAdressBookMapper.mapMuliple(domain, createUpdateDirEntry,
				mailbox == null ? Collections.emptyMap() : Map.of(uid, mailbox), locationCache,
				InstallationId.getIdentifier());
		store.batchSaveDeleteAdressBookRecord(nextVersion, createUpdate, deletedUids);
	}

	private boolean dropHiddenEntry(ItemValue<DirEntry> de) {
		return DROP_HIDDEN && de.value.hidden;
	}

	@Override
	public void rebuild() {
		synchronized (getLock(domainUid).rebuildLock()) {
			init();
			remove();
			produce();
		}
	}
}
