import {
    ACTIONS,
    CRITERIA_TARGETS,
    CRITERIA_MATCHERS,
    read,
    write,
    CRITERIA_OPERATORS
} from "../components/preferences/fields/customs/FilterRules/filterRules.js";

describe("filters", () => {
    describe("read", () => {
        test("with AND operator", () => {
            const rawFilter = {
                client: "bluemind",
                trigger: "IN",
                active: true,
                conditions: [
                    {
                        operator: "AND",
                        negate: false,
                        filter: { fields: ["from.email"], operator: "EQUALS", values: ["Toto"] },
                        conditions: []
                    },
                    {
                        operator: "AND",
                        negate: false,
                        filter: { fields: ["to.email"], operator: "EQUALS", values: ["Tata"] },
                        conditions: []
                    },
                    {
                        operator: "AND",
                        negate: false,
                        filter: { fields: ["subject"], operator: "CONTAINS", values: ["Toto&Tata"] },
                        conditions: []
                    },
                    {
                        operator: "AND",
                        negate: true,
                        filter: { fields: ["part.content"], operator: "CONTAINS", values: ["un gros mot"] },
                        conditions: []
                    },
                    {
                        operator: "AND",
                        negate: false,
                        filter: { fields: ["headers.X-My-Header"], operator: "MATCHES", values: ["head-bang"] },
                        conditions: []
                    }
                ],
                actions: [
                    {
                        name: "MOVE",
                        folder: "INBOX"
                    },
                    {
                        name: "REDIRECT",
                        emails: "toto.test@bluemind.net",
                        keepCopy: false
                    },
                    {
                        name: "MARK_AS_READ"
                    }
                ],
                name: "N/A",
                stop: true
            };
            const filter = read([rawFilter]);
            expect(filter).toMatchSnapshot();
        });

        test("read - no criteria", () => {
            const rawFilter = {
                client: "bluemind",
                trigger: "IN",
                active: true,
                conditions: [],
                actions: [
                    {
                        name: "MOVE",
                        folder: "INBOX"
                    },
                    {
                        name: "REDIRECT",
                        emails: "toto.test@bluemind.net",
                        keepCopy: false
                    },
                    {
                        name: "MARK_AS_READ"
                    }
                ],
                name: "N/A",
                stop: true
            };
            const filter = read([rawFilter]);
            expect(filter).toMatchSnapshot();
        });

        test("Mark message as important when The content contains: 'bonjour' OR 'salut' and does not contains 'qwerty' AND 'azerty'", () => {
            const rawFilter = {
                client: "bluemind",
                trigger: "IN",
                active: true,
                name: "test1",
                conditions: [
                    {
                        operator: "AND",
                        conditions: [
                            {
                                operator: "OR",
                                filter: {
                                    fields: ["part.content"],
                                    operator: "CONTAINS",
                                    comparator: "SUBSTRING",
                                    modifier: "NONE",
                                    values: ["bonjour"]
                                },
                                conditions: [],
                                negate: false
                            },
                            {
                                operator: "OR",
                                filter: {
                                    fields: ["part.content"],
                                    operator: "CONTAINS",
                                    comparator: "SUBSTRING",
                                    modifier: "NONE",
                                    values: ["salut"]
                                },
                                conditions: [],
                                negate: false
                            }
                        ],
                        negate: false
                    },
                    {
                        operator: "AND",
                        conditions: [
                            {
                                operator: "AND",
                                filter: {
                                    fields: ["part.content"],
                                    operator: "CONTAINS",
                                    comparator: "SUBSTRING",
                                    modifier: "NONE",
                                    values: ["qwerty"]
                                },
                                conditions: [],
                                negate: false
                            },
                            {
                                operator: "AND",
                                filter: {
                                    fields: ["part.content"],
                                    operator: "CONTAINS",
                                    comparator: "SUBSTRING",
                                    modifier: "NONE",
                                    values: ["azerty"]
                                },
                                conditions: [],
                                negate: false
                            }
                        ],
                        negate: true
                    }
                ],
                actions: [
                    {
                        name: "MARK_AS_IMPORTANT"
                    }
                ],
                stop: false
            };
            expect(read([rawFilter])).toEqual([
                {
                    actions: [
                        {
                            name: "MARK_AS_IMPORTANT"
                        }
                    ],
                    active: true,
                    criteriaOperator: CRITERIA_OPERATORS.OR,
                    exceptionOperator: CRITERIA_OPERATORS.AND,
                    criteria: [
                        {
                            exception: false,
                            matcher: "CONTAINS",
                            target: {
                                name: undefined,
                                type: "part.content"
                            },
                            value: "bonjour"
                        },
                        {
                            exception: false,
                            matcher: "CONTAINS",
                            target: {
                                name: undefined,
                                type: "part.content"
                            },
                            value: "salut"
                        }
                    ],
                    editable: true,
                    exceptions: [
                        {
                            exception: true,
                            matcher: "CONTAINS",
                            target: {
                                name: undefined,
                                type: "part.content"
                            },
                            value: "qwerty"
                        },
                        {
                            exception: true,
                            matcher: "CONTAINS",
                            target: {
                                name: undefined,
                                type: "part.content"
                            },
                            value: "azerty"
                        }
                    ],
                    id: undefined,
                    name: "test1",
                    terminal: false
                }
            ]);
        });
    });

    describe("write", () => {
        test("write - no criteria", () => {
            const filter = {
                actions: [
                    { name: ACTIONS.DELETE.name },
                    {
                        name: ACTIONS.FORWARD.name,
                        keepCopy: false,
                        emails: ["toto.test@bluemind.net"]
                    }
                ],
                active: true,
                criteria: [],
                exceptions: [],
                name: "MyFilter",
                terminal: true
            };
            const rawFilter = write(filter);
            expect(rawFilter).toMatchSnapshot();
        });

        test("write", () => {
            const filter = {
                actions: [
                    { name: ACTIONS.DELETE.name },
                    {
                        name: ACTIONS.FORWARD.name,
                        keepCopy: false,
                        emails: ["toto.test@bluemind.net"]
                    }
                ],
                active: true,
                criteriaOperator: CRITERIA_OPERATORS.AND,
                exceptionOperator: CRITERIA_OPERATORS.AND,
                criteria: [
                    {
                        target: { type: CRITERIA_TARGETS.TO, name: "" },
                        matcher: CRITERIA_MATCHERS.EQUALS,
                        value: "Tata",
                        exception: false
                    },
                    {
                        target: { type: CRITERIA_TARGETS.HEADER, name: "X-Machin-Chouette" },
                        matcher: CRITERIA_MATCHERS.EQUALS,
                        value: "Bidule",
                        exception: false
                    }
                ],
                exceptions: [
                    {
                        target: { type: CRITERIA_TARGETS.BODY, name: "" },
                        matcher: CRITERIA_MATCHERS.CONTAINS,
                        value: "un gros mot",
                        exception: true
                    }
                ],
                name: "MyFilter",
                terminal: true
            };
            const rawFilter = write(filter);
            expect(rawFilter).toMatchSnapshot();
        });

        test("Delete and forward mail when mail to is Tata or header 'X-Machin-Chouette' is Bidule and mail to is not Tonton", () => {
            const filter = {
                actions: [
                    { name: ACTIONS.DELETE.name },
                    {
                        name: ACTIONS.FORWARD.name,
                        keepCopy: false,
                        emails: ["toto.test@bluemind.net"]
                    }
                ],
                active: true,
                criteriaOperator: CRITERIA_OPERATORS.OR,
                exceptionOperator: CRITERIA_OPERATORS.OR,
                criteria: [
                    {
                        target: { type: CRITERIA_TARGETS.TO, name: "" },
                        matcher: CRITERIA_MATCHERS.EQUALS,
                        value: "Tata",
                        exception: false
                    },
                    {
                        target: { type: CRITERIA_TARGETS.HEADER, name: "X-Machin-Chouette" },
                        matcher: CRITERIA_MATCHERS.EQUALS,
                        value: "Bidule",
                        exception: false
                    }
                ],
                exceptions: [
                    {
                        target: { type: CRITERIA_TARGETS.TO, name: "" },
                        matcher: CRITERIA_MATCHERS.EQUALS,
                        value: "Tonton",
                        exception: false
                    }
                ],
                name: "MyFilter",
                terminal: true
            };
            const rawFilter = write(filter);
            expect(rawFilter.conditions).toEqual([
                {
                    conditions: [
                        {
                            conditions: [],
                            filter: { fields: ["to.email", "cc.email"], operator: "EQUALS", values: ["Tata"] },
                            negate: false,
                            operator: "OR"
                        },
                        {
                            conditions: [],
                            filter: { fields: ["headers.X-Machin-Chouette"], operator: "EQUALS", values: ["Bidule"] },
                            negate: false,
                            operator: "OR"
                        }
                    ],
                    negate: false,
                    operator: "AND"
                },
                {
                    conditions: [
                        {
                            conditions: [],
                            filter: { fields: ["to.email", "cc.email"], operator: "EQUALS", values: ["Tonton"] },
                            negate: false,
                            operator: "OR"
                        }
                    ],
                    negate: true,
                    operator: "AND"
                }
            ]);
        });
    });
});
