import app from "@bluemind/global.app";
import Vue2TouchEvents from "vue2-touch-events";

import { default as AlertStore, DefaultAlert } from "@bluemind/alert.store";
import { Bluetiful } from "@bluemind/bluetiful";
import NotificationManager from "@bluemind/commons/utils/notification";
import i18n, { generateDateTimeFormats, TranslationRegistry } from "@bluemind/i18n";
import { inject } from "@bluemind/inject";
import router from "@bluemind/router";
import { initSentry } from "@bluemind/sentry";
import store from "@bluemind/store";
import VueBus from "@bluemind/vue-bus";
import { extend } from "@bluemind/vuex-router";
import VueSockjsPlugin from "@bluemind/vue-sockjs";

import routes from "./routes";
import BannerL10N from "../l10n/banner/";
import registerDependencies from "./registerDependencies";
import PreferencesStore from "./preferences/store";
import RootAppL10N from "../l10n/root/";
import RootAppStore from "./rootAppStore";
import SettingsL10N from "../l10n/preferences/";
import SettingsStore from "./settingsStore";
import MainApp from "./components/MainApp";
import Command from "@bluemind/command";

// FEATBOOK-114
window["bundles"]["net.bluemind.webapp.root.js"]["start"] = () => {
    const userSession = window.bmcSessionInfos;
    registerDependencies(userSession);
    setDateTimeFormat(userSession);

    // Request permission for Notification
    NotificationManager.requestPermission();

    initSentry(userSession);

    app.use(Bluetiful, { lang: userSession.lang });
    routes.forEach(r => router.addRoute(r));
    app.use(router);

    TranslationRegistry.register(BannerL10N);
    TranslationRegistry.register(RootAppL10N);
    TranslationRegistry.register(SettingsL10N);
    app.use(i18n);

    initWebApp(userSession, app);
    app.use(VueBus, store);

    if (userSession.userId) {
        app.use(VueSockjsPlugin, { VueBus });
    }
    app.use(Vue2TouchEvents, { disableClick: true });
    app.use(Command);
    app.component("MainApp", MainApp);
    app.mount("#app");
    async function initWebApp(userSession, app) {
        initStore();

        app.component("DefaultAlert", DefaultAlert);
        app.use(store);
        if (userSession.userId) {
            setDateTimeFormat(userSession);
        }
    }

    function initStore() {
        extend(router, store);
        store.registerModule("alert", AlertStore);
        store.registerModule("root-app", RootAppStore);
        store.registerModule("settings", SettingsStore);
        store.registerModule("preferences", PreferencesStore);
    }

    async function setDateTimeFormat(session) {
        if (userSession.userId) {
            const timeformat = await inject("UserSettingsPersistence").getOne(session.userId, "timeformat");
            const dateTimeFormats = generateDateTimeFormats(timeformat);
            Object.entries(dateTimeFormats).forEach(entry => {
                i18n.global.setDateTimeFormat(entry[0], entry[1]);
            });
        }
    }
};

(async () => {
    if ("serviceWorker" in navigator) {
        try {
            const registration = await navigator.serviceWorker.register("service-worker.js");
            // eslint-disable-next-line no-console
            console.log("Registration succeeded. Scope is " + registration.scope);
        } catch (error) {
            // eslint-disable-next-line no-console
            console.log("Registration failed with " + error);
        }

        navigator.serviceWorker.addEventListener("message", event => {
            if (event.data.type === "ERROR") {
                // eslint-disable-next-line no-console
                console.warn(event.data.payload.message);
            }
        });

        navigator.serviceWorker.addEventListener("waiting", () => {
            // eslint-disable-next-line no-console
            console.warn(
                "A new service worker is installed but cannot be activated until all tabs running the current version have been fully unloaded."
            );
        });

        navigator.serviceWorker.addEventListener("installed", event => {
            if (event.isUpdate) {
                // eslint-disable-next-line no-console
                console.warn("A new version of the site is available, please refresh the page.");
            }
        });
    }
})();
