/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.backend.bm.task;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.eas.backend.Changes;
import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.MSTask;
import net.bluemind.eas.backend.bm.compat.OldFormats;
import net.bluemind.eas.backend.bm.impl.CoreConnect;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.ContentImportEntityForChange;
import net.bluemind.eas.backend.importer.ContentImportEntityForDeletion;
import net.bluemind.eas.dto.base.AirSyncBaseResponse;
import net.bluemind.eas.dto.base.AppData;
import net.bluemind.eas.dto.base.BodyType;
import net.bluemind.eas.dto.base.ChangeType;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.base.DisposableByteSource;
import net.bluemind.eas.dto.base.LazyLoaded;
import net.bluemind.eas.dto.sync.CollectionSyncRequest.Options.ConflicResolution;
import net.bluemind.eas.dto.tasks.TasksResponse;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.todolist.api.ITodoList;
import net.bluemind.todolist.api.VTodo;

public class TaskBackend extends CoreConnect {

	private TaskConverter converter;
	private final ISyncStorage storage;

	public TaskBackend(ISyncStorage storage) {
		converter = new TaskConverter();
		this.storage = storage;
	}

	public Changes getContentChanges(CollectionIdContext collectionIdContext, long version) throws ActiveSyncException {
		Changes changes = new Changes();

		try {
			HierarchyNode folder = storage.getHierarchyNode(collectionIdContext);
			ITodoList service = getService(collectionIdContext.backendSession(), folder.containerUid);

			ContainerChangeset<Long> changeset = service.changesetById(version);
			EasLogUser.logDebugAsUser(collectionIdContext.getUserLogin(), logger,
					"[{}][{}] get task changes. created: {}, updated: {}, deleted: {}, folder: {}, version: {}",
					collectionIdContext.getUserLogin(), collectionIdContext.backendSession().getDevId(),
					changeset.created.size(), changeset.updated.size(), changeset.deleted.size(), folder.containerUid,
					version);

			changes.version = changeset.version;

			for (long id : changeset.created) {
				changes.items
						.add(getItemChange(collectionIdContext.collectionId(), id, ItemDataType.TASKS, ChangeType.ADD));
			}

			for (long id : changeset.updated) {
				changes.items.add(
						getItemChange(collectionIdContext.collectionId(), id, ItemDataType.TASKS, ChangeType.CHANGE));
			}

			for (long id : changeset.deleted) {
				changes.items.add(
						getItemChange(collectionIdContext.collectionId(), id, ItemDataType.TASKS, ChangeType.DELETE));
			}

		} catch (ServerFault e) {
			if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
				EasLogUser.logWarnAsUser(collectionIdContext.getUserLogin(), logger, e.getMessage());
			} else {
				EasLogUser.logExceptionAsUser(collectionIdContext.getUserLogin(), e, logger);
			}
			changes.version = version;
		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(collectionIdContext.getUserLogin(), e, logger);
			// BM-7227
			// Something went wrong
			// Send current version number to prevent full sync
			changes.version = version;
		}

		return changes;
	}

	public CollectionItem store(ContentImportEntityForChange contentEntity) throws ActiveSyncException {
		CollectionItem ret = null;
		String user = contentEntity.backendSession.getLoginAtDomain();
		HierarchyNode folder = storage.getHierarchyNode(contentEntity.collectionIdContext);
		ITodoList service = getService(contentEntity.backendSession, folder.containerUid);
		try {
			if (contentEntity.serverId.isPresent()) {
				String serverId = contentEntity.serverId.get();
				Long id = getItemId(serverId);

				if (id != null) {
					ItemValue<VTodo> item = service.getCompleteById(id);
					if (item == null) {
						EasLogUser.logDebugAsUser(user, logger, "Fail to find VTodo {}", id);
						return CollectionItem.of(contentEntity.collectionId, id);
					}

					if (contentEntity.conflictPolicy == ConflicResolution.SERVER_WINS
							&& item.version > contentEntity.syncState.version) {
						String msg = String.format(
								"Both server (version '%d') and client (version '%d') changes. Conflict resolution is SERVER_WINS for task %s::%s",
								item.version, contentEntity.syncState.version, contentEntity.collectionId, serverId);
						throw new ActiveSyncException(msg);
					}

					VTodo oldTodo = item.value;
					VTodo todo = converter.convert(contentEntity.data);

					if (todo.description == null) {
						// GLAG-72 desc can be ghosted
						todo.description = oldTodo.description;
					}

					// Do not loose location and categories
					todo.location = oldTodo.location;
					todo.categories = oldTodo.categories;

					try {
						service.updateById(id, todo);
						ret = CollectionItem.of(contentEntity.collectionId, id);
						EasLogUser.logInfoAsUser(user, logger,
								"Update todo bs: {}, collection: {}, serverId: {}, summary: {}, completed: {}", user,
								folder.containerUid, serverId, todo.summary, todo.completed);
					} catch (Exception e) {
						EasLogUser.logErrorAsUser(user, logger, "Fail to update todo bs:" + user + ", collection: "
								+ folder.containerUid + ", serverId: " + serverId + ", summary:" + todo.summary);
					}
				}

			} else {
				VTodo event = converter.convert(contentEntity.data);
				String uid = UUID.randomUUID().toString();
				service.create(uid, event);

				ItemValue<VTodo> created = service.getComplete(uid);
				ret = CollectionItem.of(contentEntity.collectionId, created.internalId);
			}

		} catch (ServerFault e) {
			throw new ActiveSyncException(e);
		}

		return ret;
	}

	private ITodoList getService(BackendSession bs, String containerUid) throws ServerFault {
		return getTodoListService(bs, containerUid);
	}

	public void delete(ContentImportEntityForDeletion contentEntity) throws ActiveSyncException {
		if (contentEntity.serverIds != null) {
			try {
				for (CollectionItem serverId : contentEntity.serverIds) {
					HierarchyNode folder = storage.getHierarchyNode(
							new CollectionIdContext(contentEntity.backendSession, serverId.collectionId));
					ITodoList service = getService(contentEntity.backendSession, folder.containerUid);

					service.deleteById(serverId.itemId);
				}
			} catch (ServerFault e) {
				if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
					throw new ActiveSyncException(e);
				}
				EasLogUser.logExceptionAsUser(contentEntity.user, e, logger);
			}
		}

	}

	public AppData fetch(BackendSession bs, ItemChangeReference ic) throws ActiveSyncException {
		try {
			HierarchyNode folder = storage.getHierarchyNode(new CollectionIdContext(bs, ic.getServerId().collectionId));
			ITodoList service = getService(bs, folder.containerUid);

			ItemValue<VTodo> todo = service.getCompleteById(ic.getServerId().itemId);
			AppData ret = toAppData(bs, todo);

			return ret;
		} catch (Exception e) {
			throw new ActiveSyncException(e.getMessage(), e);
		}
	}

	public Map<Long, AppData> fetchMultiple(CollectionIdContext collectionIdContext, List<Long> ids)
			throws ActiveSyncException {
		HierarchyNode folder = storage.getHierarchyNode(collectionIdContext);
		ITodoList service = getService(collectionIdContext.backendSession(), folder.containerUid);

		List<ItemValue<VTodo>> todos = service.multipleGetById(ids);
		Map<Long, AppData> res = new HashMap<>(ids.size());
		todos.stream().forEach(todo -> {
			try {
				AppData data = toAppData(collectionIdContext.backendSession(), todo);
				res.put(todo.internalId, data);
			} catch (Exception e) {
				EasLogUser.logErrorExceptionAsUser(collectionIdContext.getUserLogin(), e, logger,
						"Fail to convert todo {}", todo.uid);
			}
		});

		return res;
	}

	private AppData toAppData(BackendSession bs, ItemValue<VTodo> todo) {
		MSTask msTask = converter.convert(todo.value, bs.getUser().getTimeZone());

		TasksResponse cr = OldFormats.update(msTask);
		AppData data = AppData.of(cr);

		if (msTask.description != null && !msTask.description.trim().isEmpty()) {
			final AirSyncBaseResponse airSyncBase = new AirSyncBaseResponse();
			airSyncBase.body = new AirSyncBaseResponse.Body();
			airSyncBase.body.type = BodyType.PlainText;
			airSyncBase.body.data = DisposableByteSource.wrap(msTask.description.trim());
			airSyncBase.body.estimatedDataSize = (int) airSyncBase.body.data.size();
			data.body = LazyLoaded.loaded(airSyncBase);
		}
		return data;
	}

}
