/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.backend.bm.mail;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.BinaryBody;
import org.apache.james.mime4j.dom.Body;
import org.apache.james.mime4j.dom.Multipart;
import org.apache.james.mime4j.dom.TextBody;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.message.MessageImpl;
import org.jsoup.Jsoup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.buffer.Buffer;
import net.bluemind.backend.mail.api.DispositionType;
import net.bluemind.backend.mail.api.IMailboxItems;
import net.bluemind.backend.mail.api.MailboxItem;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Part;
import net.bluemind.backend.mail.api.MessageBody.Recipient;
import net.bluemind.backend.mail.api.MessageBody.RecipientKind;
import net.bluemind.core.api.Stream;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.eas.backend.MSAttachment;
import net.bluemind.eas.backend.MSEmail;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.mime4j.common.AddressableEntity;
import net.bluemind.mime4j.common.Mime4JHelper;

public class DraftSynchronization {

	private final MailboxItem msgFromServer;
	private final String user;
	private final IMailboxItems mailboxItemService;
	private final CollectionId collectionId;
	private final long msgItemId;
	private final List<Part> attachmentPartsToRemoveAfter;

	private MSEmail msgFromDevice;
	private boolean updateDraft;

	private static Logger logger = LoggerFactory.getLogger(DraftSynchronization.class);

	private DraftSynchronization(boolean update, String user, IMailboxItems mailboxItemService, CollectionItem ci,
			ItemValue<MailboxItem> msgFromServer) {
		this.updateDraft = update;
		this.user = user;
		this.mailboxItemService = mailboxItemService;
		this.collectionId = ci.collectionId;
		this.msgItemId = msgFromServer.internalId;
		this.attachmentPartsToRemoveAfter = new ArrayList<>();

		this.msgFromServer = msgFromServer.value;
	}

	public DraftSynchronization(String user, IMailboxItems mailboxItemService, ItemValue<MailboxItem> msgFromServer,
			MSEmail email, CollectionItem ci) {
		this(false, user, mailboxItemService, ci, msgFromServer);

		this.msgFromDevice = email;

	}

	public DraftSynchronization(String user, IMailboxItems mailboxItemService, ItemValue<MailboxItem> msgFromServer,
			MSEmail email, CollectionItem ci, MessageBody msgBodyFromDevice) {
		this(true, user, mailboxItemService, ci, msgFromServer);

		this.msgFromDevice = null;

		this.msgFromServer.body = msgBodyFromDevice;
	}

	public MessageBody getMsgBodyFromServer() {
		return msgFromServer.body;
	}

	public MailboxItem getMsgFromServer() {
		return msgFromServer;
	}

	public List<Part> getAttachmentPartsToRemoveAfter() {
		return attachmentPartsToRemoveAfter;
	}

	public boolean isUpdateDraft() {
		return updateDraft;
	}

	public void injectChanges() throws IOException {
		updateDraft = false;
		updateDraft |= injectSubject();
		updateDraft |= injectRecipients(RecipientKind.Primary, msgFromDevice.getMessage().getTo());
		updateDraft |= injectRecipients(RecipientKind.CarbonCopy, msgFromDevice.getMessage().getCc());
		updateDraft |= injectRecipients(RecipientKind.BlindCarbonCopy, msgFromDevice.getMessage().getBcc());
		updateDraft |= injectAttachments();
		updateDraft |= injectUpdatedBody();
	}

	private boolean injectUpdatedBody() throws IOException {
		int msgBodySizeFromD = sizeOfMsgImplPartBody(msgFromDevice.getMessage(), Mime4JHelper.TEXT_HTML);
		int msgBodySizeFromS = sizeOfMsgBodyPartFromServer(mailboxItemService, msgItemId, Mime4JHelper.TEXT_HTML);
		if (msgBodySizeFromD == -1 || msgBodySizeFromS == -1) {
			return false;
		}

		EasLogUser.logDebugAsUser(user, logger, "Draft [{}:{}] body html size from device {} // from server {}",
				collectionId.getFolderId(), msgItemId, msgBodySizeFromD, msgBodySizeFromS);

		boolean bodySizeChanged = Math.abs(msgBodySizeFromS - msgBodySizeFromD) > 1;
		if (bodySizeChanged) {
			if (logger.isDebugEnabled()) {
				String messageBodyContentFromD = readMsgImplPartBody(msgFromDevice.getMessage(),
						Mime4JHelper.TEXT_HTML);
				EasLogUser.logDebugAsUser(user, logger, "Draft [{}:{}] body html content from device \r\n{} ",
						collectionId.getFolderId(), msgItemId, messageBodyContentFromD);
				String messageBodyContentFromS = readMsgBodyPartFromServer(mailboxItemService, msgItemId,
						Mime4JHelper.TEXT_HTML);
				EasLogUser.logDebugAsUser(user, logger, "Draft [{}:{}] body html content from server \r\n{} ",
						collectionId.getFolderId(), msgItemId, messageBodyContentFromS);
			}

			MessageBody currentBody = getMsgBodyFromServer();
			String newHtmlBody = readMsgImplPartBody(msgFromDevice.getMessage(), Mime4JHelper.TEXT_HTML);
			String newTextBody = htmlToPlain(newHtmlBody);
			String newHtmlPartAddress = mailboxItemService
					.uploadPart(VertxStream.stream(Buffer.buffer(newHtmlBody.getBytes())));
			String newTextPartAddress = mailboxItemService
					.uploadPart(VertxStream.stream(Buffer.buffer(newTextBody.getBytes())));
			injectIntoPart(currentBody.structure, Mime4JHelper.TEXT_HTML, newHtmlPartAddress);
			injectIntoPart(currentBody.structure, Mime4JHelper.TEXT_PLAIN, newTextPartAddress);
		}
		return bodySizeChanged;
	}

	private boolean injectSubject() {
		boolean subjectChanges = msgFromDevice.getMessage() != null //
				&& ((msgFromDevice.getMessage().getSubject() != null
						&& !msgFromDevice.getMessage().getSubject().equals(getMsgBodyFromServer().subject) //
						|| (msgFromDevice.getMessage().getSubject() == null
								&& getMsgBodyFromServer().subject != null)));
		if (subjectChanges) {
			getMsgBodyFromServer().subject = msgFromDevice.getMessage().getSubject();
		}
		return subjectChanges;
	}

	private boolean injectAttachments() throws IOException {
		Set<MSAttachment> attachments = msgFromDevice.getAttachments();
		if (!attachments.isEmpty()) {
			if (attachments.stream().anyMatch(a -> !a.deleted)) {
				getAttachements().forEach(a -> {
					try {
						InputStream is = readAttachmentPartStream(a);
						byte[] allBytes = is.readAllBytes();
						String partAddr = mailboxItemService.uploadPart(VertxStream.stream(Buffer.buffer(allBytes)));
						Part p = Part.create(a.getFilename(), a.getMimeType(), partAddr);
						p.dispositionType = DispositionType.ATTACHMENT;
						p.encoding = "base64";
						p.charset = "us-ascii";
						p.size = allBytes.length;
						attachmentPartsToRemoveAfter.add(p);
					} catch (IOException e) {
						EasLogUser.logErrorExceptionAsUser(user, e, logger,
								"Attachment [{}] has not been correctly upload for collectionId {} itemId {}",
								a.getFilename(), collectionId, msgItemId);
					}
				});

				getMsgBodyFromServer().structure.children.addAll(attachmentPartsToRemoveAfter);
			}

			attachments.stream() //
					.filter(attachment -> attachment.deleted) //
					.map(attachment -> extractPartAddressFromFileReference(attachment.filereference)) //
					.filter(partAddress -> partAddress > 0) //
					.forEach(partAddress -> getMsgBodyFromServer().structure.children.remove(partAddress.intValue()));
		}

		return !attachments.isEmpty();
	}

	private int extractPartAddressFromFileReference(String filereference) {
		try {
			String partAddress = filereference.replace(collectionId.getValue() + "_" + msgItemId + "_", "");
			int endIndex = partAddress.indexOf("_");
			partAddress = partAddress.substring(0, endIndex);
			return Integer.valueOf(partAddress) - 1;
		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(user, e, logger);
			return 0;
		}
	}

	private boolean injectRecipients(RecipientKind kind, AddressList msgRecipients) {
		List<Recipient> recipientsToUpdate = getMsgBodyFromServer().recipients;
		List<Recipient> filterRecipientsFromServer = recipientsToUpdate.stream().filter(r -> r.kind == kind).toList();
		if (msgRecipients == null && filterRecipientsFromServer.isEmpty()) {
			return false;
		}

		boolean recipientsChanges = (msgRecipients == null || msgRecipients.isEmpty())
				&& !filterRecipientsFromServer.isEmpty();
		if (recipientsChanges) {
			recipientsToUpdate.removeIf(r -> r.kind == kind);
		} else {
			List<Recipient> filterRecipientsFromDevice = msgRecipients.flatten().stream()
					.map(a -> Recipient.create(kind, a.getName(), a.getAddress())).toList();
			recipientsChanges = !filterRecipientsFromServer.stream().map(r -> r.address)
					.collect(Collectors.joining(","))
					.equals(filterRecipientsFromDevice.stream().map(r -> r.address).collect(Collectors.joining(",")));
			if (recipientsChanges) {
				recipientsToUpdate.removeIf(r -> r.kind == kind);
				recipientsToUpdate.addAll(filterRecipientsFromDevice);
			}
		}

		if (recipientsChanges) {
			getMsgBodyFromServer().recipients = recipientsToUpdate;
		}
		return recipientsChanges;
	}

	private List<AddressableEntity> getAttachements() throws IOException {
		if (msgFromDevice.getMessage().isMultipart()) {
			Multipart mp = (Multipart) msgFromDevice.getMessage().getBody();
			return Mime4JHelper.expandTree(mp.getBodyParts()).stream()
					.filter(p -> "attachment".equals(p.getDispositionType())).toList();
		}
		return Collections.emptyList();
	}

	private static Optional<ByteArrayOutputStream> readMsgImplPartBodyStream(MessageImpl msg, String mime)
			throws IOException {
		TextBody tBody = null;
		if (msg.getBody() instanceof TextBody) {
			tBody = (TextBody) msg.getBody();
		} else {
			Multipart mp = (Multipart) msg.getBody();
			List<Multipart> bParts = mp.getBodyParts().stream().map(b -> (Multipart) b.getBody()).toList();
			Optional<Multipart> alternativePart = bParts.stream()
					.filter(p -> "alternative".equalsIgnoreCase(p.getSubType())).findFirst();
			if (alternativePart.isEmpty()) {
				return Optional.empty();
			}
			List<AddressableEntity> parts = Mime4JHelper.expandTree(alternativePart.get().getBodyParts());
			Body body = parts.stream().filter(p -> mime.equals(p.getMimeType())).findFirst().map(p -> p.getBody())
					.orElse(null);
			if (body != null) {
				tBody = (TextBody) body;
			}
		}

		if (tBody != null) {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			tBody.writeTo(out);
			return Optional.ofNullable(out);
		}
		return Optional.empty();
	}

	private static String readMsgImplPartBody(MessageImpl msg, String mime) throws IOException {
		return readMsgImplPartBodyStream(msg, mime).map(s -> s.toString()).orElse(null);
	}

	private static int sizeOfMsgImplPartBody(MessageImpl msg, String mime) throws IOException {
		return readMsgImplPartBodyStream(msg, mime).map(s -> s.size()).orElse(-1);
	}

	private static Optional<Stream> readMsgBodyPartStreamFromServer(IMailboxItems service, long itemId, String mime) {
		ItemValue<MailboxItem> mailboxItem = service.getCompleteById(itemId);
		String partAddress = mailboxItem.value.body.structure.parts().stream().filter(p -> p.mime.equals(mime))
				.findFirst().map(part -> part.address).orElse(null);
		if (partAddress != null) {
			return Optional.ofNullable(service.fetch(mailboxItem.value.imapUid, partAddress, null, null, null, null));
		}
		return Optional.empty();
	}

	private static String readMsgBodyPartFromServer(IMailboxItems service, long itemId, String mime) {
		Optional<Stream> msgPartStream = readMsgBodyPartStreamFromServer(service, itemId, mime);
		return msgPartStream.map(s -> GenericStream.streamToString(s)).orElse(null);
	}

	private static int sizeOfMsgBodyPartFromServer(IMailboxItems service, long itemId, String mime) {
		Optional<Stream> msgPartStream = readMsgBodyPartStreamFromServer(service, itemId, mime);
		return msgPartStream.map(s -> GenericStream.streamToBytes(s).length).orElse(-1);
	}

	private static InputStream readAttachmentPartStream(AddressableEntity part) throws IOException {
		return ((BinaryBody) part.getBody()).getInputStream();
	}

	private static String htmlToPlain(String newHtmlBody) {
		return Jsoup.parse(newHtmlBody).text();
	}

	private static void injectIntoPart(Part part, String mimeType, String newPartAddress) {
		if (part.mime.equals(mimeType)) {
			part.address = newPartAddress;
		} else {
			for (Part child : part.children) {
				injectIntoPart(child, mimeType, newPartAddress);
			}
		}
	}

}
