/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.eas.backend.IBackend;
import net.bluemind.eas.backend.IContentsExporter;
import net.bluemind.eas.backend.IHierarchyExporter;
import net.bluemind.eas.backend.ISettingsExporter;
import net.bluemind.eas.backend.ISettingsImporter;
import net.bluemind.eas.backend.bm.calendar.CalendarBackend;
import net.bluemind.eas.backend.bm.calendar.CalendarsNotificationHandler;
import net.bluemind.eas.backend.bm.contacts.ContactNotificationHandler;
import net.bluemind.eas.backend.bm.contacts.ContactsBackend;
import net.bluemind.eas.backend.bm.mail.DelegationNotificationHandler;
import net.bluemind.eas.backend.bm.mail.FolderNotificationHandler;
import net.bluemind.eas.backend.bm.mail.MailBackend;
import net.bluemind.eas.backend.bm.mail.NewMailNotificationHandler;
import net.bluemind.eas.backend.bm.mail.SyncResetHandler;
import net.bluemind.eas.backend.bm.state.InternalState;
import net.bluemind.eas.backend.bm.task.TaskBackend;
import net.bluemind.eas.backend.bm.task.TaskNotificationHandler;
import net.bluemind.eas.backend.bm.user.LogoutNotificationHandler;
import net.bluemind.eas.backend.bm.user.UserBackend;
import net.bluemind.eas.backend.importer.IContentsImporter;
import net.bluemind.eas.backend.importer.IHierarchyImporter;
import net.bluemind.eas.dto.user.MSUser;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.hornetq.client.Consumer;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.Topic;
import net.bluemind.network.topology.Topology;
import net.bluemind.vertx.common.http.BasicAuthHandler;

public class BMBackend implements IBackend {

	private final IHierarchyImporter hImporter;
	private final IContentsImporter cImporter;
	private final IHierarchyExporter exporter;
	private final IContentsExporter contentsExporter;
	private ISettingsExporter settingsExporter;
	private ISettingsImporter settingsImporter;
	private final ISyncStorage store;
	private final UserBackend userBackend;

	private Consumer mailPushConsumer;
	private Consumer hierarchyNotificationsConsumer;
	private Consumer resetSyncNotificationsConsumer;
	private Consumer calPushConsumer;
	private Consumer contactPushConsumer;
	private Consumer taskPushConsumer;
	private Consumer delegationNotificationsConsumer;
	private Consumer logoutNotificationsConsumer;
	private static final Logger logger = LoggerFactory.getLogger(BMBackend.class);

	public BMBackend(ISyncStorage store) {

		FolderBackend folderBackend = new FolderBackend(store);
		MailBackend mailBackend = new MailBackend(store);
		CalendarBackend calendarBackend = new CalendarBackend(store);
		ContactsBackend contactsBackend = new ContactsBackend(store);
		TaskBackend taskBackend = new TaskBackend(store);
		this.store = store;
		this.userBackend = new UserBackend();

		hImporter = new HierarchyImporter(folderBackend);
		exporter = new HierarchyExporter(folderBackend);
		settingsExporter = new SettingsExporter(folderBackend);
		settingsImporter = new SettingsImporter(folderBackend);
		cImporter = new ContentsImporter(mailBackend, calendarBackend, contactsBackend, taskBackend);
		contentsExporter = new ContentsExporter(mailBackend, calendarBackend, contactsBackend, taskBackend);

		MQ.init(new MQ.IMQConnectHandler() {

			@Override
			public void connected() {
				setupDelegationPush();
				setupFolderPush();
				setupResetSyncPush();
				setupMailPush();
				setupCalPush();
				setupContactPush();
				setupTaskPush();
				setupLogoutPush();
			}
		});
		;
	}

	private void setupLogoutPush() {
		if (logoutNotificationsConsumer != null) {
			return;
		}
		logoutNotificationsConsumer = MQ.registerConsumer(Topic.CORE_SESSIONS, new LogoutNotificationHandler());
	}

	private void setupMailPush() {
		if (mailPushConsumer != null) {
			return;
		}
		mailPushConsumer = MQ.registerConsumer(Topic.MAPI_ITEM_NOTIFICATIONS, new NewMailNotificationHandler(store));

	}

	private void setupDelegationPush() {
		if (delegationNotificationsConsumer != null) {
			return;
		}
		delegationNotificationsConsumer = MQ.registerConsumer(Topic.MAPI_DELEGATION_NOTIFICATIONS,
				new DelegationNotificationHandler());
	}

	private void setupResetSyncPush() {
		if (resetSyncNotificationsConsumer != null) {
			return;
		}
		resetSyncNotificationsConsumer = MQ.registerConsumer(Topic.EAS_RESET_SYNC, new SyncResetHandler());
	}

	private void setupFolderPush() {
		if (hierarchyNotificationsConsumer != null) {
			return;
		}
		hierarchyNotificationsConsumer = MQ.registerConsumer(Topic.MAPI_HIERARCHY_NOTIFICATIONS,
				new FolderNotificationHandler());
	}

	private void setupCalPush() {
		if (calPushConsumer != null) {
			return;
		}
		calPushConsumer = MQ.registerConsumer(Topic.CALENDAR_NOTIFICATIONS, new CalendarsNotificationHandler(store));
	}

	private void setupContactPush() {
		if (contactPushConsumer != null) {
			return;
		}
		contactPushConsumer = MQ.registerConsumer(Topic.CONTACT_NOTIFICATIONS, new ContactNotificationHandler(store));
	}

	private void setupTaskPush() {
		if (taskPushConsumer != null) {
			return;
		}
		taskPushConsumer = MQ.registerConsumer(Topic.TASK_NOTIFICATIONS, new TaskNotificationHandler(store));
	}

	@Override
	public IHierarchyImporter getHierarchyImporter(BackendSession bs) {
		return hImporter;
	}

	@Override
	public IHierarchyExporter getHierarchyExporter(BackendSession bs) {
		return exporter;
	}

	@Override
	public IContentsImporter getContentsImporter(BackendSession bs) {
		return cImporter;
	}

	@Override
	public IContentsExporter getContentsExporter(BackendSession bs) {
		return contentsExporter;
	}

	@Override
	public ISettingsExporter getSettingsExporter(BackendSession bs) {
		return settingsExporter;
	}

	@Override
	public ISettingsImporter getSettingsImporter(BackendSession bs) {
		return settingsImporter;
	}

	@Override
	public MSUser getUser(String loginAtDomain, String sid) throws ActiveSyncException {
		return userBackend.getUser(loginAtDomain, sid);
	}

	@Override
	public String getPictureBase64(BackendSession bs, int photoId) {
		return userBackend.getPictureBase64(bs, photoId);
	}

	@Override
	public void acknowledgeRemoteWipe(BackendSession bs) {
		EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger, "Client '{}' acknowledges RemoteWipe",
				bs.getDeviceId().getIdentifier());
		// TODO do something (send email to admin? device owner?)
	}

	@Override
	public void initInternalState(BackendSession bs) {
		EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "Set internal state");

		InternalState is = new InternalState();
		is.coreUrl = "http://" + Topology.get().core().value.address() + ":8090";
		is.sid = bs.getSid();

		bs.setInternalState(is);
	}

	@Override
	public void purgeSessions() {
		logger.info("bm-core (re)started. Purge backend sessions.");
		BasicAuthHandler.purgeSessions();
		UserBackend.purgeSession();
	}

	@Override
	public void purgeSession(String sid) {
		List<String> purgeSessionsLogin = new ArrayList<>();
		purgeSessionsLogin.addAll(BasicAuthHandler.purgeSession(sid));
		purgeSessionsLogin.addAll(UserBackend.purgeSession(sid));
		if (logger.isDebugEnabled() && !purgeSessionsLogin.isEmpty()) {
			logger.debug("bm-core logout. Purged user backend sessions {} with sid: {}.", purgeSessionsLogin, sid);
		}
	}

	@Override
	public Set<String> getSubscriptions(BackendSession bs) {
		return hImporter.importUserSubscriptionOwners(bs);
	}

}
