/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cql.persistence;

import java.time.Instant;
import java.util.Date;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BatchStatement;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.DefaultBatchType;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;

import net.bluemind.cql.CqlPersistenceException;

public class CqlAbstractStore {

	protected final CqlSession session;

	@FunctionalInterface
	public interface EntityPopulator<T> {
		public int populate(Row r, int index, T value);
	}

	protected final <W> EntityPopulator<W> voidPop() {
		return (r, i, w) -> i;
	}

	@FunctionalInterface
	public interface EntityCreator<T> {
		public T create(Row r);
	}

	protected CqlAbstractStore(CqlSession s) {
		this.session = s;
	}

	private ResultSet cql(String query, Object... params) {
		var prep = session.prepare(query);
		return session.execute(prep.bind(params));
	}

	protected <W> W unique(String query, EntityCreator<W> creator, EntityPopulator<W> pop, Object... params) {
		var res = cql(query, params);
		Row r = res.one();
		if (r == null) {
			return null;
		}
		var ret = creator.create(r);
		pop.populate(r, 0, ret);
		return ret;
	}

	protected <W> List<W> map(String query, EntityCreator<W> creator, EntityPopulator<W> pop, Object... params) {
		var res = cql(query, params);
		return res.map(r -> {
			var ret = creator.create(r);
			pop.populate(r, 0, ret);
			return ret;
		}).all();
	}

	protected boolean voidCql(String query, Object... params) {
		ResultSet r = cql(query, params);
		return r.wasApplied();
	}

	protected ResultSet batch(BoundStatement... sts) {
		BatchStatement bs = BatchStatement.newInstance(DefaultBatchType.LOGGED, sts);
		return session.execute(bs);
	}

	protected BoundStatement b(String q, Object... params) {
		return session.prepare(q).bind(params);
	}

	protected static Date dateOrNull(Row r, int index) {
		Instant instant = r.getInstant(index);
		return instant == null ? null : Date.from(instant);
	}

	/**
	 * Executes the statement and check if something was applied.
	 * 
	 * @param query
	 * @param params
	 * @return
	 * @throws CqlPersistenceException if nothing was applied
	 */
	protected void applyCql(String query, Object... params) {
		var res = cql(query, params);
		if (!res.wasApplied()) {
			throw new CqlPersistenceException("Query " + query + " was not applied.");
		}
	}

}
