import { vi } from "vitest";
import Vuex from "vuex";
import { cloneDeep } from "@bluemind/commons/utils/lang";
import { ADDRESS_AUTOCOMPLETE } from "~/getters";
import { INCREASE_ADDRESS_WEIGHT, DELETE_ADDRESS_WEIGHTS, EXCLUDE_ADDRESS } from "~/mutations";
import storeData from "../addressAutocomplete";

// Vue.use(Vuex);
const RealDate = global.Date;
const Now = vi.fn(() => 0);
describe("address autocomplete", () => {
    let store;
    beforeEach(() => {
        global.Date.now = Now;
        store = new Vuex.Store(cloneDeep(storeData));
        Now.mockClear();
    });
    afterEach(() => {
        global.Date = RealDate;
    });
    describe("INCREASE_ADDRESS_WEIGHT", () => {
        test("INCREASE_ADDRESS_WEIGHT: the last increased addr is the most heavy", () => {
            const address1 = "address-1";

            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            Now.mockImplementationOnce(() => 1);
            const address2 = "address-2";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            expect(store.state.addressWeights[address2]).toBeGreaterThan(store.state.addressWeights[address1]);
        });
        test("INCREASE_ADDRESS_WEIGHT: the more you increased an addr the heavier it is", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            Now.mockImplementationOnce(() => 1);
            const address2 = "address-2";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            expect(store.state.addressWeights[address1]).toBeGreaterThan(store.state.addressWeights[address2]);
        });
        test("INCREASE_ADDRESS_WEIGHT: If you do not increase an addr for a given amout of time, the most recent addr will be heavier", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            Now.mockImplementationOnce(() => 10000000);
            const address2 = "address-2";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            expect(store.state.addressWeights[address2]).toBeGreaterThan(store.state.addressWeights[address1]);
        });
        test("INCREASE_ADDRESS_WEIGHT: There is a max bonus weight for increasing an addr", () => {
            const address1 = "address-1";
            for (let i = 0; i < 20; i++) {
                store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            }
            const address2 = "address-2";
            for (let i = 0; i < 40; i++) {
                store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            }
            expect(store.state.addressWeights[address2]).toEqual(store.state.addressWeights[address1]);
        });
    });
    describe("EXCLUDE_ADDRESS", () => {
        test("EXCLUDE_ADDRESS: An Excluded address a minimal weight", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            const address2 = "address-2";
            Now.mockImplementationOnce(() => 1);
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            store.commit(EXCLUDE_ADDRESS, { address: address2 });
            expect(store.state.addressWeights[address1]).toBeGreaterThan(store.state.addressWeights[address2]);
        });
        test("EXCLUDE_ADDRESS: An Excluded address can be re-weighted", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            const address2 = "address-2";
            Now.mockImplementationOnce(() => 1);
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            store.commit(EXCLUDE_ADDRESS, { address: address2 });
            Now.mockImplementationOnce(() => 1);
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            expect(store.state.addressWeights[address2]).toBeGreaterThan(store.state.addressWeights[address1]);
        });
    });
    describe("DELETE_ADDRESS_WEIGHTS", () => {
        test("DELETE_ADDRESS_WEIGHTS mutation", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            const address2 = "address-2";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            const address3 = "address-3";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address3 });
            const address4 = "address-4";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address4 });
            expect(Object.keys(store.state.addressWeights)?.length).toBe(4);

            store.commit(DELETE_ADDRESS_WEIGHTS);
            expect(Object.keys(store.state.addressWeights)?.length).toBeFalsy();
        });
    });
    describe("ADDRESS_AUTOCOMPLETE", () => {
        test("ADDRESS_AUTOCOMPLETE: The heavier addresses are at the bottom of the list", () => {
            const address1 = "address-1";
            expect(store.state.addressWeights[address1]).toBeFalsy();
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });

            const address2 = "address-2";
            Now.mockImplementationOnce(() => 1);
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });
            expect(store.state.addressWeights[address1]).toBeGreaterThan(store.state.addressWeights[address2]);

            const address3 = "address-3";
            Now.mockImplementationOnce(() => 2);
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address3 });
            expect(store.state.addressWeights[address3]).toBeGreaterThan(store.state.addressWeights[address2]);

            const { sortedAddresses } = store.getters[ADDRESS_AUTOCOMPLETE];
            expect(sortedAddresses[0]).toEqual(address2);
            expect(sortedAddresses[1]).toEqual(address3);
            expect(sortedAddresses[2]).toEqual(address1);
        });
        test("ADDRESS_AUTOCOMPLETE: The excluded addresses are not in the list", () => {
            const address1 = "address-1";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address1 });
            store.commit(EXCLUDE_ADDRESS, { address: address1 });

            const address2 = "address-2";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address2 });

            const address3 = "address-3";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address3 });
            store.commit(EXCLUDE_ADDRESS, { address: address3 });

            const address4 = "address-4";
            store.commit(INCREASE_ADDRESS_WEIGHT, { address: address4 });

            const { excludedAddresses, sortedAddresses } = store.getters[ADDRESS_AUTOCOMPLETE];
            expect(excludedAddresses).toHaveLength(2);
            expect(excludedAddresses[0]).toEqual(address1);
            expect(excludedAddresses[1]).toEqual(address3);
            expect(sortedAddresses).toHaveLength(2);
            expect(sortedAddresses[0]).toEqual(address2);
            expect(sortedAddresses[1]).toEqual(address4);
        });
    });
});
