import { base64ToArrayBuffer } from "@bluemind/arraybuffer";
import { cloneDeep } from "@bluemind/commons/utils/lang";
import { PartsBuilder, MimeType, CID_DATA_ATTRIBUTE } from "@bluemind/email";
import { html2text } from "@bluemind/html-utils";
import { inject } from "@bluemind/inject";
import { partUtils, messageUtils } from "@bluemind/mail";

import { UPDATE_MESSAGE_STRUCTURE } from "~/actions";
import type { ActionContext } from "vuex";
import { SET_MESSAGE_PREVIEW, SET_MESSAGES_STATUS, SET_SAVE_ERROR } from "~/mutations";

import { Actions, scheduleAction } from "./draftActionsScheduler";
import { DispositionType, type MailboxItemsClient, type MessageBody } from "@bluemind/backend.mail.api";
const { MessageStatus } = messageUtils;
const { sanitizeTextPartForCyrus } = partUtils;

type Context = ActionContext<Record<string, Message>, Record<string, Message>>;

type Message = {
    key: string;
    structure: NonNullable<MessageBody["structure"]>;
    folderRef: { uid: string };
};

type Payload = {
    message: Message;
    content: string;
};

export default async function (context: Context, payload: Payload) {
    return scheduleAction(() => setMessageContent(context, payload), Actions.SET_CONTENT, true);
}

export async function debouncedSetMessageContent(context: Context, payload: Payload) {
    return scheduleAction(() => setMessageContent(context, payload), Actions.SET_CONTENT);
}

async function setMessageContent({ commit, dispatch }: Context, { message, content }: Payload) {
    try {
        commit(SET_MESSAGE_PREVIEW, { key: message.key, preview: html2text(content) });
        const structure = await buildBodyStructureFromContent(message, content);
        const resolved = await dispatch(UPDATE_MESSAGE_STRUCTURE, { key: message.key, structure });
        commit(SET_SAVE_ERROR, null);
        return resolved;
    } catch (err) {
        console.error(err);
        commit(SET_MESSAGES_STATUS, [{ key: message.key, status: MessageStatus.SAVE_ERROR }]);
        commit(SET_SAVE_ERROR, err);
    }
}

async function buildBodyStructureFromContent(message: Message, html: string) {
    const htmlContent = sanitizeTextPartForCyrus(html);
    const textContent = sanitizeTextPartForCyrus(html2text(htmlContent));
    return buildContentPart(htmlContent, textContent, message);
}

async function buildContentPart(htmlContent: string, textContent: string, message: Message) {
    const alternativePart = await buildAlternativePart(htmlContent, textContent, message);
    const isMixed = MimeType.isMixed(message.structure);

    if (isMixed) {
        const oldStructure = cloneDeep(message.structure);
        const attachmentParts = oldStructure?.children?.filter(isAttachment) ?? [];
        const mixedPart = PartsBuilder.createMixedPart([alternativePart, ...attachmentParts]);
        return santizeMixedPart(mixedPart);
    }
    return alternativePart;
}

async function buildAlternativePart(htmlContent: string, textContent: string, message: Message) {
    const service = inject("MailboxItemsPersistence", message.folderRef.uid);
    const alternativePart = getMainAlternativePart(message.structure);
    const htmlRelatedPart = await buildHtmlRelatedPart(service, htmlContent, alternativePart);
    const textPlainPart = await buildTextPlainPart(service, textContent);
    const partsToPreserve = alternativePart ? getPartsToPreserve(alternativePart) : [];
    return PartsBuilder.createAlternativePart(textPlainPart, htmlRelatedPart, ...partsToPreserve);
}

function getPartsToPreserve(alternativePart: MessageBody.Part) {
    const parts = alternativePart.children?.filter(
        part => ![MimeType.TEXT_HTML, MimeType.TEXT_PLAIN, MimeType.MULTIPART_RELATED].includes(part.mime)
    );
    return parts ?? [];
}

function getMainAlternativePart(structure: MessageBody.Part): MessageBody.Part | undefined {
    const alternativePart = MimeType.isAlternative(structure)
        ? structure
        : structure.children?.find(MimeType.isAlternative);

    return cloneDeep(alternativePart);
}

async function buildTextPlainPart(service: MailboxItemsClient, textContent: string) {
    const address = await service.uploadPart(textContent);
    return PartsBuilder.createTextPart(address);
}

async function buildHtmlRelatedPart(
    service: MailboxItemsClient,
    htmlContent: string,
    alternativePart?: MessageBody.Part
) {
    const { images, cids, html } = extractInlineImages(htmlContent);
    const oldRelatedPart = alternativePart?.children?.find((part: MessageBody.Part) => MimeType.isRelated(part));
    const relatedPart = await buildRelatedPart(service, oldRelatedPart, cids, images, html);
    return sanitizeRelatedPart(relatedPart);
}

async function buildNewHtmlPart(service: MailboxItemsClient, html: string) {
    const address = await service.uploadPart(html);
    return PartsBuilder.createHtmlPart(address);
}

async function buildRelatedPart(
    service: MailboxItemsClient,
    oldRelatedPart: MessageBody.Part | undefined,
    cids: Set<string>,
    images: Record<string, InlineImage>,
    html: string
) {
    const htmlPart = await buildNewHtmlPart(service, html);
    const imageParts = oldRelatedPart?.children?.filter(part => !isObsoleteImage(part, cids)) ?? [];
    const newImageParts = await uploadNewImages(service, oldRelatedPart, images);
    return PartsBuilder.createRelatedPart([htmlPart, ...imageParts, ...newImageParts]);
}

type InlineImage = {
    data: ArrayBuffer;
    mime: string;
    size: number;
    name?: string;
};

async function uploadNewImages(
    service: MailboxItemsClient,
    relatedPart: MessageBody.Part | undefined,
    images: Record<string, InlineImage>
) {
    const children = relatedPart?.children || [];
    const existingCids = children.map(({ contentId }) => contentId);

    const newImageParts: MessageBody.Part[] = [];
    for (const cid in images) {
        if (!existingCids.includes(cid) && images[cid]?.data) {
            const { data, mime, size, name } = images[cid];
            const blob = new Blob([data], { type: mime });
            const address = await service.uploadPart(blob);
            const part = PartsBuilder.createInlinePart({ mime, size, address, cid, name });
            newImageParts.push(part);
        }
    }
    return newImageParts;
}

function sanitizeRelatedPart(relatedPart: MessageBody.Part): MessageBody.Part {
    if (relatedPart.children?.length === 1) {
        return relatedPart.children.pop()!;
    }
    return relatedPart;
}

function santizeMixedPart(mixedPart: MessageBody.Part) {
    if (mixedPart.children?.length === 1 && MimeType.isMultipart(mixedPart.children[0])) {
        return mixedPart.children.pop()!;
    }
    return mixedPart;
}

function isObsoleteImage(part: MessageBody.Part, cids: Set<string>): boolean {
    return !cids.has(part.contentId!);
}

function extractInlineImages(htmlContent: string) {
    const htmlContentDocument = new DOMParser().parseFromString(htmlContent, "text/html");
    const imageNodes = Array.from(htmlContentDocument.querySelectorAll("img[src]"));
    const images: Record<string, InlineImage> = {};
    const cids = new Set<string>();

    imageNodes.forEach(node => {
        const img = node as HTMLImageElement;
        const cid = extractCid(img);
        const src = node.getAttribute("src");
        if (cid && src) {
            try {
                if (!cids.has(cid) && isBase64Image(src)) {
                    images[cid] = extractPartFromDataUrl(img, src);
                }
                cids.add(cid);
                node.setAttribute("src", `cid:${cid.slice(1, -1)}`);
            } catch {
                // ignore invalid image
            }
        }
    });
    return { images, cids, html: htmlContentDocument.body.innerHTML };
}

function isBase64Image(src: string): boolean {
    return src.startsWith("data:image");
}

function extractCid(imageNode: HTMLImageElement): string | undefined {
    return imageNode.getAttribute(CID_DATA_ATTRIBUTE) || undefined;
}

function extractPartFromDataUrl(imageNode: HTMLImageElement, src: string): InlineImage {
    const { data, metadata, name } = extractDataFromImg(src, imageNode.getAttribute("alt") || undefined);
    if (data && metadata) {
        return {
            data,
            mime: getMimeType(metadata),
            size: data.byteLength,
            name
        };
    }
    throw new Error("Failed to parse image data for " + src);
}

function extractDataFromImg(src: string, alt?: string) {
    const extractDataRegex = /data:image(.*)base64,/g;
    const metadata = src.match(extractDataRegex)?.[0] ?? "";
    const data = src.replace(metadata, "");
    return { data: convertData(data), metadata, name: alt };
}

function convertData(b64Data = "") {
    const sanitized = sanitizeB64(b64Data);
    return base64ToArrayBuffer(sanitized);
}

function sanitizeB64(base64: string) {
    return base64.replaceAll("%0A", "").replace(/[^a-zA-Z0-9+/=]/g, "");
}

function getMimeType(metadata: string): string {
    const withoutData = metadata?.replace("data:", "");
    return withoutData?.substring(0, withoutData.indexOf(";")) || "image/png";
}

function isAttachment(part: MessageBody.Part) {
    return part.dispositionType === DispositionType.ATTACHMENT;
}
