import i18n from "@bluemind/i18n";
import { computeUnit } from "@bluemind/file-utils";
import DateRanges from "./DateRanges";
import { messageUtils, loadingStatusUtils } from "@bluemind/mail";
const { LoadingStatus } = loadingStatusUtils;
const { isScheduled } = messageUtils;

export function byDate(conversations) {
    const rangeByKey = {};
    const allDateRanges = new DateRanges();
    const currentDateRanges = [];

    conversations.forEach(conversation => {
        if (conversation.date) {
            const dateRange = getDateRange(conversation, allDateRanges, currentDateRanges);
            if (dateRange) {
                rangeByKey[conversation.key] = i18n.global.t(
                    dateRange.i18n,
                    dateRange.date && {
                        date: i18n.global.d(dateRange.date, dateRange.dateFormat)
                    }
                );
            }
        }
    });

    return rangeByKey;
}

export function bySize(conversations) {
    const rangeByKey = {};
    const allSizeRanges = [
        { max: 1999 },
        { min: 2000, max: 9999 },
        { min: 10000, max: 99999 },
        { min: 100000, max: 999999 },
        { min: 1000000 }
    ];
    const currentSizeRanges = [];

    conversations.forEach(({ key, size }) => {
        if (size) {
            const sizeRange = allSizeRanges.find(({ min, max }) => (!min || min <= size) && (!max || max >= size));
            if (sizeRange && !currentSizeRanges.includes(sizeRange)) {
                const pluralizationCount = sizeRange.min ? (sizeRange.max ? 1 : 2) : 0;
                rangeByKey[key] = i18n.global.tc("mail.list.range.size", pluralizationCount, {
                    min: computeUnit(sizeRange.min, i18n.global),
                    max: computeUnit(sizeRange.max, i18n.global)
                });
                currentSizeRanges.push(sizeRange);
            }
        }
    });

    return rangeByKey;
}

export function bySender(conversations) {
    const rangeByKey = {};
    const currentSenders = [];

    conversations.forEach(({ key, from }) => {
        const { address, dn } = from || { address: "", dn: "" };
        if (address && !currentSenders.includes(address)) {
            rangeByKey[key] = dn ? `${dn} <${address}>` : address;
            currentSenders.push(address);
        }
    });

    return rangeByKey;
}

export function outbox(conversations, messages) {
    const rangeByKey = {};
    conversations.forEach(key => {
        const message = messages[key];
        if (message?.loading === LoadingStatus.LOADED) {
            const swimlaneStatus = getSwimlaneStatus(message);
            if (!Object.values(rangeByKey).includes(swimlaneStatus)) {
                rangeByKey[key] = swimlaneStatus;
            }
        }
    });
    return rangeByKey;
}

function getDateRange(conversation, allDateRanges, currentDateRanges) {
    const dateRange = allDateRanges.sortedArray.find(dateRange =>
        dateRange.contains(typeof conversation.date === "number" ? conversation.date : conversation.date.getTime())
    );
    if (!currentDateRanges.includes(dateRange)) {
        currentDateRanges.push(dateRange);
        return dateRange;
    }
}

function getSwimlaneStatus(message) {
    const swimlaneStatus = {
        sendScheduled: i18n.global.t("mail.list.range.send_scheduled"),
        sending: i18n.global.t("mail.list.range.sending")
    };

    return isScheduled(message) ? swimlaneStatus.sendScheduled : swimlaneStatus.sending;
}
