import { markRaw } from "vue";
import { inject } from "@bluemind/inject";
import { EmailExtractor } from "@bluemind/email";
import { searchVCardsHelper } from "@bluemind/contact";
import { messageUtils } from "@bluemind/mail";
import { AlertTypes, useDispatchAlert } from "@bluemind/bluetiful";
import ForwardedEventAlert from "./ForwardedEventAlert.vue";
const { isEventRequest } = messageUtils;

export default {
    props: {
        message: {
            type: Object,
            required: true
        }
    },
    setup() {
        const { dispatchAlert, removeAlert } = useDispatchAlert();
        return { dispatchAlert, removeAlert };
    },
    computed: {
        eventOrganizer() {
            return this.$store.state.mail.consultPanel?.currentEvent?.serverEvent?.value?.main?.organizer;
        },
        organizerCannotBeFound() {
            return this.eventOrganizer == null;
        },
        sender() {
            return this.message.from;
        },
        organizerHasSameEmailAddressAsSender() {
            return Boolean(this.eventOrganizer?.mailto) && this.eventOrganizer.mailto === this.sender.address;
        },
        alert() {
            return {
                uid: `MESSAGE_IMIP_FORWARDED_${this.message.key}`,
                renderer: {
                    component: markRaw(ForwardedEventAlert),
                    props: {
                        alert: {
                            payload: {
                                organizer: this.eventOrganizer?.commonName,
                                sender: this.message.from.dn
                            }
                        }
                    }
                },
                type: AlertTypes.INFO
            };
        }
    },
    watch: {
        eventOrganizer: {
            async handler() {
                if (
                    isEventRequest(this.message) &&
                    (await this.eventIsForwarded()) &&
                    !this.message.eventInfo.isResourceBooking
                ) {
                    this.dispatchAlert(this.alert);
                } else {
                    this.removeAlert(this.alert);
                }
            },
            immediate: true
        }
    },

    unmounted() {
        this.removeAlert(this.alert);
    },

    methods: {
        async eventIsForwarded() {
            if (this.organizerCannotBeFound || this.organizerHasSameEmailAddressAsSender) {
                return false;
            }
            return !(await this.isOrganizerAnAliasForSender());
        },
        async isOrganizerAnAliasForSender() {
            const searchResults = await this.retreiveContactsByEmailAddress(
                EmailExtractor.extractEmail(this.sender.address)
            );
            return searchResults.some(hasAlias(this.extractUidFromDir(this.eventOrganizer?.dir), this.sender.address));
        },
        async retreiveContactsByEmailAddress(emailAddress) {
            const searchResults = await inject("AddressBooksPersistence").search(
                searchVCardsHelper(emailAddress, { size: 50 })
            );
            return searchResults.values;
        },
        extractUidFromDir(dir) {
            if (!dir) {
                return "";
            }
            return dir.split("/").pop();
        }
    },

    render() {
        return "";
    }
};

function hasAlias(organiserUid, senderEmailAddress) {
    return result => result.uid === organiserUid && result.value.mail === senderEmailAddress;
}
