/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.directory;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.core.utils.JsonUtils.ValueWriter;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.GenerationIndex;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.common.backup.RestorableDirEntry;
import net.bluemind.dataprotect.service.BackupPath;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.Member;
import net.bluemind.mailshare.api.Mailshare;
import net.bluemind.network.topology.Topology;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class DirectoryBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/directory");
	private static final Logger logger = LoggerFactory.getLogger(DirectoryBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	private static final TypeReference<RestorableDirEntry<Mailshare>> RESTORABLE_DIRENTRY_MAILSHARE_TYPE = new TypeReference<RestorableDirEntry<Mailshare>>() {
	};
	private static final TypeReference<RestorableDirEntry<ResourceDescriptor>> RESTORABLE_DIRENTRY_RESOURCE_TYPE = new TypeReference<RestorableDirEntry<ResourceDescriptor>>() {
	};

	public DirectoryBackupRepository(RepositoryBackupPath repositoryPath) {
		this.repositoryPath = repositoryPath;
	}

	public DirectoryBackupRepository(Path repositoryPath) {
		this(new RepositoryBackupPath(repositoryPath));
	}

	private static PartGeneration getGenPart(List<PartGeneration> parts) {
		return Optional
				.ofNullable(parts.stream().filter(p -> p.getWorkerDataType() == WorkerDataType.DIRECTORY).findFirst()
						.orElse(parts.stream().filter(p -> p.getWorkerDataType() == WorkerDataType.DIRECTORY_LEGACY)
								.findFirst().orElse(null)))
				.orElseThrow(() -> new ServerFault("Unable to find backup part 'mailshare'"));
	}

	public static DirectoryBackupRepository getRepository(DataProtectGeneration backup) {
		try {
			PartGeneration corepart = getGenPart(backup.parts);
			ItemValue<Server> coreServer = Topology.get().core();
			RepositoryBackupPath backupPath = new RepositoryBackupPath(
					Paths.get(BackupPath.get(coreServer, TagDescriptor.bm_core.getTag()), String.valueOf(corepart.id),
							"var/backups/bluemind/directory"));
			if (backupPath.exists()) {
				return new DirectoryBackupRepository(backupPath);
			}
		} catch (Exception e) {
			logger.error("Unable to get backup repository: {}", e.getMessage());
		}
		return null;
	}

	private boolean checkRepositoryPath() {
		boolean exists = repositoryPath.exists();
		if (!exists) {
			logger.error("Path {} does not exists", repositoryPath.rootPath());
		}

		return exists;
	}

	private Optional<RestorableDirEntry<Mailshare>> readZipFilesForRestorableMailshare(String domainUid,
			String entryUid) throws Exception {
		return repositoryPath.getZipFilePathForDirEntry(domainUid, entryUid).map(zipFilePath -> {
			List<RestorableDirEntry> readMailshares = extractDirEntriesDataFromZip(entryUid, zipFilePath,
					JsonUtils.reader(RESTORABLE_DIRENTRY_MAILSHARE_TYPE));
			return readMailshares.size() == 1 ? readMailshares.getFirst() : null;
		});
	}

	private Optional<RestorableDirEntry<Mailshare>> readJsonFilesForRestorableMailshare(String entryUid)
			throws IOException {
		List<RestorableDirEntry<Mailshare>> list = (List<RestorableDirEntry<Mailshare>>) JsonReadWrite.read(
				JsonUtils.reader(RESTORABLE_DIRENTRY_MAILSHARE_TYPE), repositoryPath.rootPath(),
				p -> p.getFileName().toString().equals(entryUid + RepositoryBackupPath.JSON_FILE_EXTENSION));
		return Optional.ofNullable(list != null ? list.getFirst() : null);
	}

	public Optional<RestorableDirEntry<Mailshare>> getRestorableMailshare(String domainUid, String entryUid)
			throws Exception {
		if (!checkRepositoryPath()) {
			return Optional.empty();
		}

		Optional<RestorableDirEntry<Mailshare>> restorableMailshare = readZipFilesForRestorableMailshare(domainUid,
				entryUid);
		if (restorableMailshare.isEmpty()) {
			restorableMailshare = readJsonFilesForRestorableMailshare(entryUid);
		}
		return restorableMailshare;

	}

	private Optional<RestorableDirEntry<ResourceDescriptor>> readZipFilesForRestorableResource(String domainUid,
			String entryUid) throws Exception {
		return repositoryPath.getZipFilePathForDirEntry(domainUid, entryUid).map(zipFilePath -> {
			List<RestorableDirEntry> readResources = extractDirEntriesDataFromZip(entryUid, zipFilePath,
					JsonUtils.reader(RESTORABLE_DIRENTRY_RESOURCE_TYPE));
			return readResources.size() == 1 ? readResources.getFirst() : null;
		});
	}

	private Optional<RestorableDirEntry<ResourceDescriptor>> readJsonFilesForRestorableResource(String entryUid)
			throws IOException {
		List<RestorableDirEntry<ResourceDescriptor>> list = (List<RestorableDirEntry<ResourceDescriptor>>) JsonReadWrite
				.read(JsonUtils.reader(RESTORABLE_DIRENTRY_RESOURCE_TYPE), repositoryPath.rootPath(),
						p -> p.getFileName().toString().equals(entryUid + RepositoryBackupPath.JSON_FILE_EXTENSION));
		return Optional.ofNullable(list != null ? list.getFirst() : null);
	}

	public Optional<RestorableDirEntry<ResourceDescriptor>> getRestorableResource(String domainUid, String entryUid)
			throws Exception {
		if (!checkRepositoryPath()) {
			return Optional.empty();
		}

		Optional<RestorableDirEntry<ResourceDescriptor>> restorableResource = readZipFilesForRestorableResource(
				domainUid, entryUid);
		if (restorableResource.isEmpty()) {
			restorableResource = readJsonFilesForRestorableResource(entryUid);
		}
		return restorableResource;
	}

	private Optional<RestorableUser> readZipFilesForRestorableUser(String domainUid, String entryUid) throws Exception {
		return repositoryPath.getZipFilePathForDirEntry(domainUid, entryUid).map(zipFilePath -> {
			List<RestorableUser> readUsers = extractDirEntriesUserDataFromZip(entryUid, zipFilePath);
			return readUsers.size() == 1 ? readUsers.getFirst() : null;
		});
	}

	private Optional<RestorableUser> readJsonFilesForRestorableUser(String entryUid) throws IOException {
		ValueReader<RestorableUser> reader = JsonUtils.reader(RestorableUser.class);
		List<RestorableUser> list = (List<RestorableUser>) JsonReadWrite.read(reader, repositoryPath.rootPath(),
				p -> p.getFileName().toString().equals(entryUid + RepositoryBackupPath.JSON_FILE_EXTENSION));
		return Optional.ofNullable(list != null ? list.getFirst() : null);
	}

	/**
	 * Returns the RestorableUser available in the backup
	 *
	 * @param entryUid
	 * @return RestorableUser
	 * @throws IOException
	 */
	public Optional<RestorableUser> getRestorableUser(String domainUid, String entryUid) throws Exception {
		if (!checkRepositoryPath()) {
			return Optional.empty();
		}

		Optional<RestorableUser> restorableUser = readZipFilesForRestorableUser(domainUid, entryUid);
		if (restorableUser.isEmpty()) {
			restorableUser = readJsonFilesForRestorableUser(entryUid);
		}
		return restorableUser;
	}

	private List<RestorableUser> extractDirEntriesUserDataFromZip(String folderName, Path zipFilePath) {
		List<RestorableUser> users = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteArray -> {
				try {
					users.add((RestorableUser) JsonReadWrite.readByteArr(JsonUtils.reader(RestorableUser.class),
							byteArray));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return users;
	}

	private List<RestorableDirEntry> extractDirEntriesDataFromZip(String folderName, Path zipFilePath,
			ValueReader valueReader) {
		List<RestorableDirEntry> entries = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteBuf -> {
				try {
					entries.add((RestorableDirEntry) JsonReadWrite.readByteArr(valueReader, byteBuf));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return entries;
	}

	record BackupDirEntryRecord(Optional<GenerationIndex> genIndex, RestorableDirEntry restDirEntry, Kind kind) {

	}

	public record IndexedEntries(String version, List<GenerationIndex> entries) {

	}

	public void writeDomainIndexedEntries(String domainUid, IndexedEntries rde) throws IOException {
		Path p = repositoryPath.getStoragePath("___" + domainUid);
		JsonReadWrite.write(JsonUtils.writer(IndexedEntries.class), p, rde);
	}

	private Path getStorageDirEntryPath(RestorableDirEntry rde) {
		return repositoryPath.getStoragePath(rde.dirEntry.uid);
	}

	public String writeResource(ItemValue<DirEntry> dirEntry, ItemValue<ResourceDescriptor> resource,
			List<GenerationIndex> indexedDirEntries) throws IOException {
		RestorableDirEntry<ResourceDescriptor> rde = new RestorableDirEntry<>(dirEntry, resource);
		Path storageDirEntryPath = getStorageDirEntryPath(rde);
		indexedDirEntries.add(rde.writeRestorable(JsonUtils.writer(RestorableDirEntry.class), storageDirEntryPath));
		return storageDirEntryPath.getFileName().toString();
	}

	public String writeUser(RestorableUser rde, List<GenerationIndex> indexedDirEntries) throws IOException {
		Path storageDirEntryPath = getStorageDirEntryPath(rde);
		indexedDirEntries.add(rde.writeRestorable(JsonUtils.writer(RestorableUser.class), storageDirEntryPath));
		return storageDirEntryPath.getFileName().toString();
	}

	public String writeMailshare(ItemValue<DirEntry> dirEntry, ItemValue<Mailshare> mailshare,
			List<GenerationIndex> indexedDirEntries) throws IOException {
		RestorableDirEntry<Mailshare> rde = new RestorableDirEntry<>(dirEntry, mailshare);
		Path storageDirEntryPath = getStorageDirEntryPath(rde);
		indexedDirEntries.add(rde.writeRestorable(JsonUtils.writer(RestorableDirEntry.class), storageDirEntryPath));
		return storageDirEntryPath.getFileName().toString();
	}

	public String writeGroup(ItemValue<DirEntry> dirEntry, ItemValue<Group> group, List<Member> members,
			List<GenerationIndex> indexedDirEntries) throws IOException {
		RestorableGroup rde = new RestorableGroup(dirEntry, group, members);
		Path storageDirEntryPath = getStorageDirEntryPath(rde);
		indexedDirEntries.add(rde.writeRestorable(JsonUtils.writer(RestorableGroup.class), storageDirEntryPath));
		return storageDirEntryPath.getFileName().toString();
	}

	public void writeDirEntry(DirEntry rde, List<GenerationIndex> indexedDirEntries) {
		GenerationIndex genIndex = new GenerationIndex(rde.entryUid, rde.displayName, rde.email, rde.path, rde.kind,
				rde.dataLocation);
		indexedDirEntries.add(genIndex);
	}

	private ValueWriter getWriter(Kind kind) {
		switch (kind) {
		case USER: {
			return JsonUtils.writer(RestorableUser.class);
		}
		case RESOURCE: {
			return JsonUtils.writer(RESTORABLE_DIRENTRY_RESOURCE_TYPE.getType());
		}
		case MAILSHARE: {
			return JsonUtils.writer(RESTORABLE_DIRENTRY_MAILSHARE_TYPE.getType());
		}
		case GROUP: {
			return JsonUtils.writer(RestorableGroup.class);
		}
		default:
			throw new IllegalArgumentException("Unexpected value: " + kind);
		}
	}

}
