/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lmtp.filter.imip;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.delivery.lmtp.common.LmtpAddress;
import net.bluemind.delivery.lmtp.common.ResolvedBox;
import net.bluemind.imip.parser.IMIPInfos;
import net.bluemind.imip.parser.IMIPInfos.IMIPType;
import net.bluemind.imip.parser.ITIPMethod;
import net.bluemind.mailbox.api.Mailbox;

public abstract class IMIPHandlerFactory {

	private static final Logger logger = LoggerFactory.getLogger(IMIPHandlerFactory.class);

	interface ImipHandlerFactory {

		public IIMIPHandler get(ITIPMethod method, ResolvedBox recip, LmtpAddress sender);

		public boolean isReadOnly(IMIPInfos imip);
	}

	public static IIMIPHandler get(IMIPInfos imip, ResolvedBox recip, LmtpAddress sender) {
		return getFactory(imip).get(imip.method, recip, sender);
	}

	public static boolean isReadOnly(IMIPInfos imip) {
		return getFactory(imip).isReadOnly(imip);
	}

	private static ImipHandlerFactory getFactory(IMIPInfos imip) {
		IMIPType type = imip.type();
		if (null != type) {
			return switch (type) {
			case VEVENT -> new EventIMIPHandlerFactory();
			case VTODO -> new TodoIMIPHandlerFactory();
			};
		}
		logger.warn("IMIP infos contain no type info");
		return new UnknownIMIPHandlerFactory();
	}

	private static class EventIMIPHandlerFactory implements ImipHandlerFactory {

		@Override
		public IIMIPHandler get(ITIPMethod method, ResolvedBox recip, LmtpAddress sender) {
			if (recip.mbox.value.type != Mailbox.Type.user && recip.mbox.value.type != Mailbox.Type.resource) {
				return new NoopEventHandler();
			}
			return switch (method) {
			case REQUEST -> new EventRequestHandler(recip, sender);
			case REPLY -> new EventReplyHandler(recip, sender);
			case CANCEL -> new EventCancelHandler(recip, sender);
			case DECLINECOUNTER -> new EventDeclineCounterHandler(recip, sender);
			case COUNTER -> new EventCounterHandler(recip, sender);
			case ADD, PUBLISH, REFRESH -> new NoopEventHandler();
			default -> throw new IllegalArgumentException("Unsupported IMIP method " + method);
			};
		}

		@Override
		public boolean isReadOnly(IMIPInfos imip) {
			return imip.method == ITIPMethod.ADD || imip.method == ITIPMethod.PUBLISH
					|| imip.method == ITIPMethod.REFRESH;
		}
	}

	private static class TodoIMIPHandlerFactory implements ImipHandlerFactory {

		public IIMIPHandler get(ITIPMethod method, ResolvedBox recip, LmtpAddress sender) {
			return switch (method) {
			case REQUEST -> new TodoRequestHandler(recip, sender);
			case REPLY -> new TodoReplyHandler(recip, sender);
			case CANCEL -> new TodoCancelHandler(recip, sender);
			case ADD, COUNTER, DECLINECOUNTER, PUBLISH, REFRESH -> new NoopEventHandler();
			default -> throw new IllegalArgumentException("Unsupported IMIP method " + method);
			};
		}

		public boolean isReadOnly(IMIPInfos imip) {
			return imip.method == ITIPMethod.ADD || imip.method == ITIPMethod.COUNTER
					|| imip.method == ITIPMethod.DECLINECOUNTER || imip.method == ITIPMethod.PUBLISH
					|| imip.method == ITIPMethod.REFRESH;
		}
	}

	private static class UnknownIMIPHandlerFactory implements ImipHandlerFactory {

		@Override
		public IIMIPHandler get(ITIPMethod method, ResolvedBox recip, LmtpAddress sender) {
			return new NoopEventHandler();
		}

		@Override
		public boolean isReadOnly(IMIPInfos imip) {
			return true;
		}

	}
}