package net.bluemind.system.service.hot.internal;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.metrics.alerts.api.AlertLevel;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTask;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskExecutionMode;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskFilter;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskStatus;
import net.bluemind.system.api.hot.upgrade.IInternalHotUpgrade;
import net.bluemind.system.service.hot.HotUpgradeOperation;

public class InternalHotUpgrade extends HotUpgrade implements IInternalHotUpgrade {

	private static final Logger logger = LoggerFactory.getLogger(InternalHotUpgrade.class);

	private static Semaphore executionSemaphore = new Semaphore(1);

	private final BmContext context;
	private final Map<String, HotUpgradeOperation> operationsByName;
	private final HotUpgradeOperationExecutor operationExecutor;

	public InternalHotUpgrade(BmContext context, List<HotUpgradeOperation> operations) {
		super(context);
		this.context = context;
		this.operationsByName = operations.stream().collect(Collectors.toMap(HotUpgradeOperation::name, o -> o));
		this.operationExecutor = new HotUpgradeOperationExecutor(runningTasks);
	}

	@Override
	public void create(HotUpgradeTask task) {
		try {
			task.status = HotUpgradeTaskStatus.PLANNED;
			if (task.mandatory) {
				task.retryCount = 0;
			}
			store.create(task);
		} catch (Exception e) {
			throw new ServerFault("Unable to create task " + task, e);
		}
	}

	@Override
	public void update(HotUpgradeTask task) {
		try {
			store.updateStatus(task);
		} catch (SQLException e) {
			throw new ServerFault("Unable to update task " + task, e);
		}
	}

	@Override
	public TaskRef start(boolean onlyReady, HotUpgradeTaskExecutionMode mode) {
		return locked(semaphore -> {
			HotUpgradeTaskFilter plannedAndFailed = HotUpgradeTaskFilter
					.filter(HotUpgradeTaskStatus.PLANNED, HotUpgradeTaskStatus.FAILURE).onlyRetryable(true)
					.onlyReady(onlyReady).mode(mode);
			return context.provider().instance(ITasksManager.class).run(logger, monitor -> {
				try {
					List<HotUpgradeTask> tasks = safeList(plannedAndFailed);
					if (hasMandatoryUnreadyTask(tasks, onlyReady)) {
						monitor.end(false, "Cannot start HotUpgrades. Mandatory unready task with status FAILURE found",
								Long.toString(-1), Level.ERROR);
						semaphore.release();
						return CompletableFuture.completedFuture(null);
					}
					monitor.begin(tasks.size(), "Starting " + tasks.size() + " hot upgrade tasks", Level.INFO);
					return executeTasks(tasks, monitor).thenAccept(failureCount -> {
						String message = (failureCount == 0) ? "Hot upgrade execution successfull"
								: "Hot upgrade failed on " + failureCount + " non-mandatory tasks";
						monitor.end(failureCount == 0, message, Long.toString(failureCount), Level.INFO);
						semaphore.release();
					}).exceptionally(e -> {
						monitor.end(false, "Hot upgrade execution failed:\n" + e.getMessage(), Long.toString(-1),
								Level.ERROR);
						logger.error("Hot upgrade execution exception", e);
						semaphore.release();
						return null;
					});
				} catch (Exception e) {
					monitor.end(false, "Hot upgrade execution failed:\n" + e.getMessage(), Long.toString(-1),
							Level.ERROR);
					logger.error("Hot upgrade execution exception", e);
					semaphore.release();
					return CompletableFuture.failedFuture(e);
				}
			});
		});
	}

	@Override
	public TaskRef startLimited(long maxDuration, HotUpgradeTaskExecutionMode mode) {
		return locked(semaphore -> {
			ITasksManager taskManager = context.provider().instance(ITasksManager.class);
			return taskManager.run(this.getClass().getName() + System.currentTimeMillis(), monitor -> {
				HotUpgradeTaskFilter plannedAndFailedJobs = HotUpgradeTaskFilter
						.filter(HotUpgradeTaskStatus.PLANNED, HotUpgradeTaskStatus.FAILURE).onlyRetryable(true)
						.onlyReady(true).mode(mode);
				List<HotUpgradeTask> tasks = safeList(plannedAndFailedJobs);
				if (hasMandatoryUnreadyTask(tasks, true)) {
					monitor.end(false, "Cannot start HotUpgrades. Mandatory unready task with status FAILURE found",
							Long.toString(-1), Level.ERROR);
					semaphore.release();
					return CompletableFuture.completedFuture(null);
				}
				monitor.begin(1, "Starting " + tasks.size() + " tasks with a timelimit of " + maxDuration + " millis",
						Level.INFO);
				TreeMap<String, List<HotUpgradeTask>> collect = tasks.stream()
						.collect(groupingBy(HotUpgradeTask::groupName, TreeMap::new, toList()));
				return handleTasks(maxDuration, monitor, collect).thenRun(() -> {
					monitor.progress(1, "");
					monitor.end(true, "job hotupgrade execution finished", "", Level.INFO);
					semaphore.release();
				}).exceptionally(e -> {
					monitor.progress(1, "");
					monitor.end(false, "job hotupgrade execution finished with errors: " + e.getMessage(), "",
							Level.ERROR);
					semaphore.release();
					return null;
				});
			});
		});
	}

	private TaskRef locked(Function<Semaphore, TaskRef> execute) {
		return (executionSemaphore.tryAcquire()) ? execute.apply(executionSemaphore) : null;
	}

	private boolean hasMandatoryUnreadyTask(List<HotUpgradeTask> tasks, boolean onlyReady) {
		long now = System.currentTimeMillis();
		return tasks.stream()
				.anyMatch(t -> t.mandatory && t.status == HotUpgradeTaskStatus.FAILURE && unready(now, t, onlyReady));
	}

	private boolean unready(long now, HotUpgradeTask t, boolean onlyReady) {
		if (!onlyReady) {
			return false;
		}
		return t.updatedAt.getTime() + (t.retryDelaySeconds * 1000) > now;
	}

	private CompletableFuture<?> handleTasks(long maxDuration, IServerTaskMonitor monitor,
			TreeMap<String, List<HotUpgradeTask>> collect) {
		AtomicBoolean firstTask = new AtomicBoolean(true);
		long start = System.currentTimeMillis();
		CompletableFuture<List<HotUpgradeTask>> currentFuture = CompletableFuture.completedFuture(new ArrayList<>());
		for (String group : collect.keySet()) {
			monitor.log("Starting hotupgrade group " + group + " containing " + collect.get(group).size() + " tasks",
					Level.INFO);
			for (HotUpgradeTask task : collect.get(group)) {
				monitor.log("Starting execution of hotupgrade group " + group, Level.INFO);
				currentFuture = currentFuture.thenCompose((ret) -> {
					if (hasFailureOnMandatoryTasks(ret)) {
						return CompletableFuture.failedFuture(new Exception(
								"Mandatory task " + ret.get(0).operation + " failed! Aborting Hotupgrade execution"));
					}
					monitor.log("Starting hotupgrade task " + task, Level.INFO);
					HotUpgradeOperation operation = operationsByName.get(task.operation);
					long taskEstimatedTime;
					try {
						taskEstimatedTime = operation.estimatedTime(task);
					} catch (Exception e) {
						monitor.log("Cannot estimate time for hotupgrade task " + task, e);
						failTask(task, monitor);
						return CompletableFuture.completedStage(null);
					}
					monitor.log("Estimating " + taskEstimatedTime + "ms for hotupgrade task " + task, Level.INFO);

					if (System.currentTimeMillis() - start + taskEstimatedTime > maxDuration && !firstTask.get()) {
						monitor.log("Time limit reached. Stopping hotupgrade tasks", Level.INFO);
						return CompletableFuture.completedStage(null);
					}
					firstTask.set(false);
					return operationExecutor.get(operation, store).execute(Arrays.asList(task), monitor);
				});

			}
		}

		return currentFuture;
	}

	private void failTask(HotUpgradeTask task, IServerTaskMonitor monitor) {
		try {
			task.failed();
			store.updateStatus(task);
		} catch (SQLException e) {
			monitor.log("Unable to update task " + task, e);
		}
	}

	private List<HotUpgradeTask> safeList(HotUpgradeTaskFilter plannedAndFailed) {
		try {
			return list(plannedAndFailed);
		} catch (ServerFault e) {
			logger.warn(
					"Unable to list hot upgrade tasks from database. It's ok if it happens during the installation process");
			return Collections.emptyList();
		}
	}

	private CompletableFuture<Long> executeTasks(List<HotUpgradeTask> tasks, IServerTaskMonitor monitor) {
		Map<String, List<HotUpgradeTask>> groups = tasks.stream()
				.collect(groupingBy(HotUpgradeTask::groupName, TreeMap::new, toList()));
		List<CompletableFuture<List<HotUpgradeTask>>> operations = new ArrayList<>();
		CompletableFuture<List<HotUpgradeTask>> sequentialExecution = CompletableFuture
				.completedFuture(new ArrayList<>());

		for (Entry<String, List<HotUpgradeTask>> groupedTasks : groups.entrySet()) {
			sequentialExecution = sequentialExecution.thenCompose(v -> {
				if (hasFailureOnMandatoryTasks(v)) {
					return CompletableFuture.failedFuture(new Exception(
							"Mandatory task " + v.get(0).operation + " failed! Aborting Hotupgrade execution"));
				}

				HotUpgradeOperation operation = operationsByName.get(groupedTasks.getValue().get(0).operation);
				if (operation == null) {
					logger.error(
							"Hot upgrade task operation name '{}' doesn't match any HotUpgradeOperation class. Aborting.",
							groupedTasks.getValue().get(0).operation);
					throw new RuntimeException("Unknown hot upgrade operation name");
				}
				IServerTaskMonitor operationMonitor = monitor.subWork(operation.name(), 1);
				List<HotUpgradeTask> operationTasks = groupedTasks.getValue();

				monitor.log("Hot upgrade " + groupedTasks.getKey() + ": " + operationTasks.size() + " tasks to perform",
						Level.INFO);
				CompletableFuture<List<HotUpgradeTask>> executeOperationTasks = executeOperationTasks(operation,
						operationTasks, operationMonitor);
				operations.add(executeOperationTasks);
				return executeOperationTasks;
			});

		}

		return sequentialExecution.thenApply(ret -> {
			return operations.stream().flatMap(taskList -> {
				try {
					return taskList.get().stream().filter(t -> t.status == HotUpgradeTaskStatus.FAILURE);
				} catch (Exception e) {
					monitor.log("Execution error while evaluating hot upgrade results: " + e.getMessage(),
							AlertLevel.CRITICAL);
					logger.warn("Execution error while evaluating hot upgrade results", e);
					return Stream.empty();
				}
			}).count();
		});

	}

	private boolean hasFailureOnMandatoryTasks(List<HotUpgradeTask> tasks) {
		return tasks.stream().anyMatch(t -> t.mandatory && t.status == HotUpgradeTaskStatus.FAILURE);
	}

	private CompletableFuture<List<HotUpgradeTask>> executeOperationTasks(HotUpgradeOperation operation,
			List<HotUpgradeTask> tasks, IServerTaskMonitor monitor) {
		return (operation == null)
				? CompletableFuture
						.completedFuture(tasks.stream().map(HotUpgradeTask::failed).collect(Collectors.toList()))
				: operationExecutor.get(operation, store).execute(tasks, monitor);
	}

}
