
import { 
    TagRef
} from "@bluemind/tag.api";
/**
 * The vcard contains contact information, typically pertaining to a single
 * contact or group of contacts.
 */
export type VCard = {
    kind: VCard.Kind;
    /**
     * To identify the source of directory information contained in the content
     * type.
     */
    source?: string;
    identification?: VCard.Identification;
    deliveryAddressing?: Array<VCard.DeliveryAddressing>;
    communications?: VCard.Communications;
    organizational?: VCard.Organizational;
    explanatory?: VCard.Explanatory;
    security?: VCard.Security;
    related?: VCard.Related;
};
export namespace VCard {
    /**
     * These properties are concerned with the security of communication pathways or
     * access to the vCard.
     */
    export type Security = {
        /**
         * To specify public keys or authentication certificates associated with the
         * object that the vCard represents
         */
        keys: Array<VCard.Security.Key>;
    };
    export namespace Security {
        export type Key = VCard.BasicAttribute & {
        };

    }
    /**
     * To specify a relationship the individual this vCard represents has with
     * another.
     */
    export type Related = {
        spouse?: string;
        manager?: string;
        assistant?: string;
    };
    /**
     * To specify the kind of object the vCard represents.
     */
    export type Kind =  "group" |  "individual" |  "location" |  "org";
    export const Kind : {
        /**
         * for a vCard representing a group of persons or entities.
         */
        readonly group: Kind,
        /**
         * for a vCard representing a single person or entity.
         */
        readonly individual: Kind,
        /**
         * for a named geographical place
         */
        readonly location: Kind,
        /**
         * for a vCard representing an organization.
         */
        readonly org: Kind
    };
    /**
     * These types are used to capture information associated with the
     * identification and naming of the entity associated with the vCard.
     */
    export type Identification = {
        formatedName?: VCard.Identification.FormatedName;
        name?: VCard.Identification.Name;
        nickname?: VCard.Identification.Nickname;
        photo?: boolean;
        photoBinary?: ArrayBuffer;
        birthday?: number;
        anniversary?: number;
        gender?: VCard.Identification.Gender;
    };
    export namespace Identification {
        /**
         * To specify the text corresponding to the nickname of the object the vCard
         * represents.
         */
        export type Nickname = VCard.BasicAttribute & {
        };
        /**
         * To specify the formatted text corresponding to the name of the object the
         * vCard represents.
         */
        export type FormatedName = VCard.BasicAttribute & {
        };
        /**
         * To specify the components of the sex and gender identity of the object the
         * vCard represents.
         */
        export type Gender = VCard.BasicAttribute & {
            text?: string;
        };
        /**
         * To specify the components of the name of the object the vCard represents.
         */
        export type Name = VCard.BasicAttribute & {
            familyNames?: string;
            givenNames?: string;
            additionalNames?: string;
            prefixes?: string;
            suffixes?: string;
        };

    }
    export type BasicAttribute = {
        parameters?: Array<VCard.Parameter>;
        value: string;
    };
    /**
     * These properties are concerned with additional explanations, such as that
     * related to informational notes or revisions specific to the vCard.
     */
    export type Explanatory = {
        /**
         * To specify a uniform resource locator associated with the object to which the
         * vCard refers. Examples for individuals include personal web sites, blogs, and
         * social networking site identifiers.
         */
        urls?: Array<VCard.Explanatory.Url>;
        /**
         * To specify application category information about the vCard, also known as
         * "tags".
         */
        categories?: Array<TagRef>;
        /**
         * To specify supplemental information or a comment that is associated with the
         * vCard.
         */
        note?: string;
    };
    export namespace Explanatory {
        export type Url = VCard.BasicAttribute & {
        };

    }
    export type Parameter = {
        label: string;
        value: string;
    };
    /**
     * These properties are concerned with information associated with
     * characteristics of the organization or organizational units of the object
     * that the vCard represents.
     */
    export type Organizational = {
        /**
         * To specify the position or job of the object the vCard represents.
         */
        title?: string;
        /**
         * To specify the function or part played in a particular situation by the
         * object the vCard represents.
         */
        role?: string;
        org?: VCard.Organizational.Org;
        /**
         * To include a member in the group this vCard represents.
         */
        member?: Array<VCard.Organizational.Member>;
    };
    export namespace Organizational {
        /**
         * To specify the organizational name and units associated with the vCard.
         */
        export type Org = {
            company?: string;
            division?: string;
            department?: string;
        };
        export type Member = {
            commonName?: string;
            mailto?: string;
            containerUid?: string;
            itemUid?: string;
        };

    }
    /**
     * These types are concerned with information related to the delivery addressing
     * or label for the vCard object.
     */
    export type DeliveryAddressing = {
        address?: VCard.DeliveryAddressing.Address;
    };
    export namespace DeliveryAddressing {
        /**
         * To specify the components of the delivery address for the vCard object.
         */
        export type Address = VCard.BasicAttribute & {
            postOfficeBox?: string;
            extentedAddress?: string;
            streetAddress?: string;
            locality?: string;
            region?: string;
            postalCode?: string;
            countryName?: string;
        };

    }
    /**
     * These properties describe information about how to communicate with the
     * object the vCard represents.
     */
    export type Communications = {
        tels?: Array<VCard.Communications.Tel>;
        emails?: Array<VCard.Communications.Email>;
        impps?: Array<VCard.Communications.Impp>;
        langs?: Array<VCard.Communications.Lang>;
    };
    export namespace Communications {
        /**
         * To specify the URI for instant messaging and presence protocol communications
         * with the object the vCard represents.
         */
        export type Impp = VCard.BasicAttribute & {
        };
        /**
         * To specify the electronic mail address for communication with the object the
         * vCard represents.
         */
        export type Email = VCard.BasicAttribute & {
        };
        /**
         * To specify the language(s) that may be used for contacting the entity
         * associated with the vCard.
         */
        export type Lang = VCard.BasicAttribute & {
        };
        /**
         * To specify the telephone number for telephony communication with the object
         * the vCard represents.
         */
        export type Tel = VCard.BasicAttribute & {
            ext?: string;
        };

    }

}
