package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.FolderCounters;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.IMailboxFoldersByOwner;
import net.bluemind.backend.mail.api.ImportMailboxItemSet;
import net.bluemind.backend.mail.api.ImportMailboxItemsStatus;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxFolderSearchQuery;
import net.bluemind.backend.mail.api.SearchResult;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.api.utils.MailIndexQuery;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.Ack;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemChangelog;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.ISharedContainerStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.index.MailIndexActivator;
import net.bluemind.repository.provider.RepositoryProvider;

public class VisibilityLimitedImapReplicatedMailboxesService
		implements IMailboxFolders, IMailboxFoldersByContainer, IMailboxFoldersByOwner {
	private static final Logger logger = LoggerFactory.getLogger(VisibilityLimitedImapReplicatedMailboxesService.class);
	private BmContext context;
	private ImapReplicatedMailboxesService basicService;
	private Container container;
	private ContainerStoreService<MailboxReplica> storeService;
	private DataLocation dataLocation;
	private RBACManager rbac;

	public VisibilityLimitedImapReplicatedMailboxesService(BmContext ctx, Container cont, DataLocation dataLocation,
			ImapReplicatedMailboxesService bs, ContainerStoreService<MailboxReplica> containerService,
			RBACManager rbac) {
		this.container = cont;
		this.context = ctx;
		this.basicService = bs;
		this.storeService = containerService;
		this.dataLocation = dataLocation;
		this.rbac = rbac;

	}

	private List<ItemValue<MailboxReplica>> visibilityLimited() {
		Set<ItemValue<MailboxReplica>> visibilityLimitedSet = new HashSet<>();
		List<BaseContainerDescriptor> sharedFolders = Collections.emptyList();
		Set<String> sharedFolderUids = new HashSet<>();
		try {
			sharedFolders = getSharedMailboxReplicas();
			sharedFolderUids = sharedFolders.stream().map(s -> IMailReplicaUids.uniqueId(s.uid))
					.collect(Collectors.toSet());
		} catch (SQLException e) {
			logger.error(e.getMessage());
		}

		Map<String, ItemValue<MailboxReplica>> mbxReplicaMap = storeService.all().stream()
				.collect(Collectors.toMap(r -> r.uid, r -> r));

		// Must retrieve parent folder
		// replica.uid
		for (BaseContainerDescriptor folder : sharedFolders) {
			String referenceFolderUid = IMailReplicaUids.uniqueId(folder.uid);
			ItemValue<MailboxReplica> mbxReplica = mbxReplicaMap.get(referenceFolderUid);

			if (mbxReplica != null) {
				builParentsdHierarchy(sharedFolderUids, mbxReplicaMap, mbxReplica, visibilityLimitedSet,
						referenceFolderUid);
			}

		}
		return visibilityLimitedSet.stream().toList();
	}

	private List<BaseContainerDescriptor> getSharedMailboxReplicas() throws SQLException {
		ContainerQuery query = ContainerQuery.ownerAndType(container.owner, IMailReplicaUids.MAILBOX_RECORDS);
		IContainers containerService = context.provider().instance(IContainers.class);
		List<ContainerDescriptor> containers = containerService.all(query);
		List<Long> ids = containers.stream().map(c -> c.internalId).toList();
		ISharedContainerStore sharedContainerStore = RepositoryProvider.instance(ISharedContainerStore.class, context,
				dataLocation);

		return sharedContainerStore.mget(ids).stream()
				.filter(result -> (result.acl().verb.equals(Verb.Read) || result.acl().verb.equals(Verb.Write))
						&& (context.getSecurityContext().getMemberOf().contains(result.acl().subject)
								|| result.acl().subject.equals(context.getSecurityContext().getOwnerPrincipal())))
				.map(result -> result.desc()).toList();
	}

	private void builParentsdHierarchy(final Set<String> sharedFolderUids,
			final Map<String, ItemValue<MailboxReplica>> map, ItemValue<MailboxReplica> currItem,
			Set<ItemValue<MailboxReplica>> limitedSet, String refFolderUid) {
		currItem.value.virtualFolder = true;

		// Shared folders must not be flagged as virtuals
		if (sharedFolderUids.contains(currItem.uid) || refFolderUid.equals(currItem.uid)) {
			currItem.value.virtualFolder = false;
		}

		if (!limitedSet.contains(currItem)) {
			limitedSet.add(currItem);
		}
		if (currItem.value.parentUid == null) {
			return;
		}
		ItemValue<MailboxReplica> parent = map.get(currItem.value.parentUid);
		if (parent == null) {
			return;
		}
		builParentsdHierarchy(sharedFolderUids, map, parent, limitedSet, refFolderUid);
	}

	@Override
	public List<ItemValue<MailboxFolder>> all() {
		rbac.check(Verb.Visible.name());
		return visibilityLimited().stream().map(this::adapt).toList();
	}

	private ItemValue<MailboxFolder> adapt(ItemValue<MailboxReplica> rec) {
		if (rec == null) {
			return null;
		}
		return ItemValue.create(rec, rec.value);
	}

	@Override
	public Ack updateById(long id, MailboxFolder value) {
		throw notOnLimitedService();
	}

	private ServerFault notOnLimitedService() {
		throw new ServerFault("Operation not authorized for " + context.getSecurityContext().getOwnerPrincipal());
	}

	@Override
	public ItemIdentifier createForHierarchy(long hierId, MailboxFolder value) {
		throw notOnLimitedService();
	}

	@Override
	public ItemIdentifier createBasic(MailboxFolder value) {
		throw notOnLimitedService();
	}

	@Override
	public void deleteById(long id) {
		throw notOnLimitedService();
	}

	@Override
	public void deepDelete(long id) {
		throw notOnLimitedService();
	}

	@Override
	public ItemValue<MailboxFolder> getCompleteById(long id) {
		return visibilityLimited().stream().filter(i -> i.internalId == id).findAny().map(this::adapt).orElse(null);
	}

	public ItemValue<MailboxReplica> getCompleteByIdRepl(long id) {
		return visibilityLimited().stream().filter(i -> i.internalId == id).findAny().orElse(null);
	}

	@Override
	public List<ItemValue<MailboxFolder>> multipleGetById(List<Long> ids) {
		Map<Long, ItemValue<MailboxReplica>> indexed = visibilityLimited().stream()
				.collect(Collectors.toMap(i -> i.internalId, i -> i));
		return ids.stream().map(indexed::get).filter(Objects::nonNull).map(this::adapt).toList();
	}

	@Override
	public ItemValue<MailboxFolder> root() {
		return basicService.root();
	}

	@Override
	public ItemValue<MailboxFolder> trash() {
		return basicService.trash();
	}

	@Override
	public ItemValue<MailboxFolder> byName(String name) {
		throw notOnLimitedService();

	}

	@Override
	public ItemChangelog itemChangelog(String itemUid, Long since) throws ServerFault {
		throw notOnLimitedService();

	}

	@Override
	public ItemValue<MailboxFolder> getComplete(String uid) {
		return visibilityLimited().stream().filter(i -> i.uid.equals(uid)).findAny().map(this::adapt).orElse(null);
	}

	private ItemValue<MailboxReplica> getCompleteRepl(String uid) {
		return visibilityLimited().stream().filter(i -> i.uid.equals(uid)).findAny().orElse(null);
	}

	@Override
	public SearchResult searchItems(MailboxFolderSearchQuery query) {
		if (query.query.scope != null && query.query.scope.folderScope != null
				&& query.query.scope.folderScope.folderUid != null) {
			String folderUid = query.query.scope.folderScope.folderUid;
			ItemValue<MailboxReplica> inScope = getCompleteRepl(folderUid);
			if (inScope == null || inScope.value.virtualFolder) {
				return SearchResult.noResult();
			} else {
				return basicService.searchItems(query);
			}
		}

		List<String> folders = visibilityLimited().stream().filter(f -> !f.value.virtualFolder).map(f -> f.uid)
				.toList();
		MailIndexQuery indexQuery = MailIndexQuery.folderQuery(query, folders);
		return MailIndexActivator.getService().searchItems(container.domainUid, container.owner, indexQuery);
	}

	@Override
	public ContainerChangeset<String> changeset(Long since) throws ServerFault {
		ContainerChangeset<String> changeset = storeService.changeset(since, Long.MAX_VALUE);
		List<ItemValue<MailboxReplica>> visibilityLimited = visibilityLimited();

		return limitChangeset(changeset, visibilityLimited, e -> e.uid, this::getCompleteRepl);

	}

	@Override
	public ContainerChangeset<Long> changesetById(Long since) throws ServerFault {
		ContainerChangeset<Long> changeset = storeService.changesetById(since, Long.MAX_VALUE);
		List<ItemValue<MailboxReplica>> visibilityLimited = visibilityLimited();

		return limitChangeset(changeset, visibilityLimited, e -> e.internalId, this::getCompleteByIdRepl);

	}

	@Override
	public ContainerChangeset<ItemVersion> filteredChangesetById(Long since, ItemFlagFilter filter) throws ServerFault {
		ContainerChangeset<ItemVersion> changeset = storeService.changesetById(since, filter);
		List<ItemValue<MailboxReplica>> visibilityLimited = visibilityLimited();

		return limitChangeset(changeset, visibilityLimited, e -> new ItemVersion(e.internalId, e.version, e.updated),
				e -> getCompleteByIdRepl(e.id));

	}

	@FunctionalInterface
	private interface ToChangesetEntry<T> {
		T asEntry(ItemValue<MailboxReplica> entry);

	}

	private <T> ContainerChangeset<T> limitChangeset(ContainerChangeset<T> changeset,
			List<ItemValue<MailboxReplica>> visibleFolders, ToChangesetEntry<T> toChangesetEntry,
			Function<T, ItemValue<MailboxReplica>> getMailReplica) {

		// Must put into allChanges.deleted all elements not belonging to shared
		// hierarchy
		Set<T> allUids = Stream.of(changeset.created, changeset.updated, changeset.deleted).flatMap(Collection::stream)
				.collect(Collectors.toSet());

		if (changeset.created != null) {
			changeset.created = changesetWithParentFolders(changeset.created, visibleFolders, toChangesetEntry,
					getMailReplica).stream().toList();
			allUids.removeAll(changeset.created);
		}

		if (changeset.updated != null) {
			changeset.updated = changesetWithParentFolders(changeset.updated, visibleFolders, toChangesetEntry,
					getMailReplica).stream().toList();
			allUids.removeAll(changeset.updated);
		}

		if (changeset.deleted != null) {
			changeset.deleted.addAll(allUids);
		}

		return changeset;

	}

	private <T> Set<T> changesetWithParentFolders(List<T> changeset, List<ItemValue<MailboxReplica>> visibleFolders,
			ToChangesetEntry<T> toChangesetEntry, Function<T, ItemValue<MailboxReplica>> getMailReplica) {

		Set<T> visible = visibleFolders.stream().map(toChangesetEntry::asEntry).collect(Collectors.toSet());
		List<T> limited = changeset.stream().filter(visible::contains).toList();
		// Must get parents for to update shared folders
		Set<T> parentsList = new HashSet<>();

		limited.forEach(u -> {
			parentsList.addAll(getParentsHierarchy(visibleFolders, getMailReplica.apply(u), toChangesetEntry::asEntry));
		});
		return Stream.of(limited, parentsList).flatMap(Collection::stream).collect(Collectors.toSet());

	}

	private <T> Set<T> getParentsHierarchy(List<ItemValue<MailboxReplica>> list, ItemValue<MailboxReplica> current,
			Function<ItemValue<MailboxReplica>, T> mapper) {
		Map<String, ItemValue<MailboxReplica>> map = list.stream().collect(Collectors.toMap(r -> r.uid, r -> r));
		Set<ItemValue<MailboxReplica>> set = new HashSet<>();
		parent(map, set, current);
		return set.stream().map(mapper::apply).collect(Collectors.toSet());
	}

	private void parent(Map<String, ItemValue<MailboxReplica>> map, Set<ItemValue<MailboxReplica>> build,
			ItemValue<MailboxReplica> current) {
		if (map.get(current.uid) == null) {
			return;
		}
		ItemValue<MailboxReplica> mailboxReplica = map.get(current.uid);
		if (mailboxReplica.value.parentUid != null && map.containsKey(mailboxReplica.value.parentUid)) {
			ItemValue<MailboxReplica> parent = map.get(mailboxReplica.value.parentUid);
			build.add(parent);
			parent(map, build, parent);
		}
	}

	@Override
	public long getVersion() throws ServerFault {
		return basicService.getVersion();
	}

	@Override
	public void emptyFolder(long id) {
		removeMessages(id);
	}

	@Override
	public void removeMessages(long id) {
		ItemValue<MailboxReplica> inScope = getCompleteByIdRepl(id);
		if (inScope != null && !inScope.value.virtualFolder) {
			RBACManager folderRbac = RBACManager.forContext(context)
					.forContainer(IMailReplicaUids.mboxRecords(inScope.uid));
			folderRbac.check(Verb.Write.name());
			basicService.emptyFolder(id, false);
		} else {
			throw new ServerFault("Cannot remove messages in '" + id + "'", ErrorCode.PERMISSION_DENIED);
		}
	}

	@Override
	public void markFolderAsRead(long id) {
		ItemValue<MailboxReplica> inScope = getCompleteByIdRepl(id);
		if (inScope != null && !inScope.value.virtualFolder) {
			RBACManager folderRbac = RBACManager.forContext(context)
					.forContainer(IMailReplicaUids.mboxRecords(inScope.uid));
			folderRbac.check(Verb.Write.name());
			basicService.flag(adapt(inScope), MailboxItemFlag.System.Seen);
		}
	}

	@Override
	public ImportMailboxItemsStatus importItems(long folderDestinationId, ImportMailboxItemSet mailboxItems)
			throws ServerFault {
		throw notOnLimitedService();
	}

	@Override
	public List<ItemValue<MailboxFolder>> childrensOf(String folderUid) {
		throw notOnLimitedService();
	}

	@Override
	public List<FolderCounters> counters(List<Long> folderItemIds) {
		List<ItemValue<MailboxReplica>> visibilityLimited = visibilityLimited();
		Set<Long> idx = visibilityLimited.stream().map(i -> i.internalId).collect(Collectors.toSet());
		List<Long> filtered = folderItemIds.stream().filter(idx::contains).toList();
		return basicService.counters(filtered);
	}

	@Override
	public String getFullName(String uid) {
		throw notOnLimitedService();
	}

}
