/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.ISharedContainerStore;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlSharedContainerStore extends CqlAbstractStore implements ISharedContainerStore {

	public CqlSharedContainerStore(CqlSession s) {
		super(s);
	}

	@Override
	public List<SharedContainerStoreResult> mget(List<Long> uids) throws SQLException {

		List<Long> reduced = map("select container_id from t_container_acl where container_id IN ?", r -> r.getLong(0),
				voidPop(), uids).stream().distinct().toList();

		List<String> uidsToLoad = map("select uid from t_container_by_id where id in ?", r -> r.getString(0), voidPop(),
				reduced);

		List<BaseContainerDescriptor> contDesc = map("""
				SELECT uid, name, owner, type, domain_uid, id
				FROM t_container
				WHERE uid IN ?
				""", r -> {
			BaseContainerDescriptor bcd = BaseContainerDescriptor.create(r.getString(0), r.getString(1), r.getString(2),
					r.getString(3), r.getString(4), false);
			bcd.internalId = r.getLong(5);
			return bcd;
		}, voidPop(), uidsToLoad);
		return contDesc.stream().flatMap(bcd -> {
			// we could load this earlier to only load containers with ACLs set on them
			List<AccessControlEntry> acls = map(
					"select subject, verbs from t_container_acl where container_id=?", r -> AccessControlEntry
							.create(r.getString(0), Verb.valueOf(r.getSet(1, String.class).iterator().next())),
					voidPop(), bcd.internalId);
			return acls.stream().map(ace -> new SharedContainerStoreResult(bcd, ace));
		}).toList();
	}

	private record ContIdAndSub(long cont, String subject) {

	}

	@Override
	public List<SharedContainerStoreResult> getForSubject(String subject) throws SQLException {
		List<ContIdAndSub> aclMaps = map("select container_id, subject FROM t_container_acl",
				r -> new ContIdAndSub(r.getLong(0), r.getString(1)), voidPop());
		List<Long> matchingConts = aclMaps.stream().filter(i -> subject.equals(i.subject())).map(i -> i.cont())
				.toList();
		return mget(matchingConts).stream().filter(s -> s.acl().subject.equals(subject)).toList();

	}

}
