/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.common.telemetry;

import java.util.Collections;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.opentelemetry.api.GlobalOpenTelemetry;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Context;
import io.opentelemetry.context.propagation.TextMapGetter;

public class EmailTracer {
	public static final String TRACE_HEADER_NAME = "X-BM-TraceId";
	private static final Logger logger = LoggerFactory.getLogger(EmailTracer.class);

	public static Context extractContext(String headerValue) {
		Map<String, String> carrierMap = Collections.singletonMap("traceparent", headerValue);
		return GlobalOpenTelemetry.get().getPropagators().getTextMapPropagator().extract(Context.current(), carrierMap,
				new TextMapGetter<Map<String, String>>() {
					@Override
					public Iterable<String> keys(Map<String, String> carrier) {
						return carrier.keySet();
					}

					@Override
					public String get(Map<String, String> carrier, String key) {
						return carrier.get(key);
					}
				});
	}

	public static void trace(String traceName, String traceparent, String messageId, String subject) {
		Context context = extractContext(traceparent);
		Span span = GlobalOpenTelemetry.getTracer(traceName) //
				.spanBuilder(traceName) //
				.setParent(context) //
				.startSpan();
		if (messageId != null) {
			span.setAttribute("email.message-id", messageId);
		}
		if (subject != null) {
			span.setAttribute("email.subject", subject);
		}
		logger.debug("header found span {}", span);
		span.end();
	}
}
