/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.mailbox.deserializer.RestoreSdsMailbox;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class MailboxBackupWorker extends MailBackupWorker {
	private static final String ZIP_FILE_PREFIX = "mailbox";

	public MailboxBackupWorker(IDPContext ctx, Path workingFolder, Path jsonIndex, RestoreSdsMailbox lastBackup,
			PartGeneration partGen) {
		super(ctx, workingFolder, jsonIndex, lastBackup, partGen);
	}

	@Override
	public MailboxIndexTotal partitionTreatment(String domainUid, String zipHash, List<String> partition) {
		List<MailboxIndexJson> mailboxesToBackup = new ArrayList<>();
		String zipFileName = CommonBackupWorker.zipFilename(ZIP_FILE_PREFIX, zipHash);
		partition.forEach(uid -> {
			MailboxIndexJson backupSdsUser = backupSdsUser(uid);
			if (backupSdsUser != null) {
				backupSdsUser.zipFileName(zipFileName);
				mailboxesToBackup.add(backupSdsUser);
			}
		});
		return new MailboxIndexTotal(mailboxesToBackup);
	}

	private MailboxIndexJson backupSdsUser(String uid) {
		DirEntry diruser = serviceProvider.instance(IDirectory.class, domain.uid).findByEntryUid(uid);
		return backupSdsUser(diruser, domain, downloadEmailContent);
	}

	public MailboxIndexJson backupSdsUser(DirEntry de, ItemValue<Domain> domain, boolean downloadEmailContent) {
		try {
			logger.info("backup single mailbox ({})", de.email);

			IUser userApi = serviceProvider.instance(IUser.class, domain.uid);
			ItemValue<User> user = userApi.getComplete(de.entryUid);
			String filename = String.format("%s@%s.json", user.value.login, domain.value.defaultAlias);

			IMailboxes mailboxApi = serviceProvider.instance(IMailboxes.class, domain.uid);
			ItemValue<Mailbox> mailbox = mailboxApi.getComplete(user.uid);
			Path outputPath = Paths.get(workingFolder.toAbsolutePath().toString(),
					String.format("%s@%s.json", mailbox.value.name, domain.value.defaultAlias));
			CyrusPartition part = CyrusPartition.forServerAndDomain(mailbox.value.dataLocation, domain.uid);
			ISdsSyncStore productionStore = productionStores.get(user.value.dataLocation);

			IDbReplicatedMailboxes mailboxapi = serviceProvider.instance(IDbReplicatedMailboxes.class, part.name,
					"user." + mailbox.uid.replace('.', '^'));
			generateSdsMailboxJson(outputPath, domain, mailbox.uid, user.value.login, mailbox.value, mailboxapi.all(),
					productionStore, downloadEmailContent);
			return new MailboxIndexJson(filename, mailbox.uid, domain.uid);
		} catch (Exception e) {
			logError(e, "Unable to backup user mailbox {}", de.entryUid);
		}
		return null;
	}

}
