/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.collect.Lists;

import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.config.Token;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupFolder;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupMailbox;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupMessage;
import net.bluemind.dataprotect.mailbox.deserializer.RestoreSdsMailbox;
import net.bluemind.dataprotect.sdsspool.SdsDataProtectSpool;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;

public class MailBackupWorker implements IMailBackupWorker {
	private static final int PARTITIONS = 32;

	protected IDPContext ctx;

	protected static Logger logger = LoggerFactory.getLogger(MailBackupWorker.class);
	protected final Path workingFolder;
	protected final Path jsonIndex;
	protected final RestoreSdsMailbox lastBackup;
	protected final IServiceProvider serviceProvider;
	protected final PartGeneration partGen;

	protected SystemConf sysconf;

	protected Map<String, ISdsSyncStore> productionStores = new HashMap<>();
	protected SdsDataProtectSpool backupStore = null;

	protected boolean downloadEmailContent;
	protected ItemValue<Domain> domain;

	public MailBackupWorker(IDPContext ctx, Path workingFolder, Path jsonIndex, RestoreSdsMailbox lastBackup,
			PartGeneration partGen) {
		this.ctx = ctx;
		this.workingFolder = workingFolder;
		this.jsonIndex = jsonIndex;
		this.lastBackup = lastBackup;
		this.partGen = partGen;
		this.serviceProvider = SPResolver.get().resolve(Token.admin0());
		this.sysconf = serviceProvider.instance(ISystemConfiguration.class).getValues();

	}

	protected void logError(Throwable throwable) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(throwable, throwable.getMessage());
	}

	protected void logError(Exception ex, String message, Object... args) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(ex, message, args);
	}

	protected record MailboxIndexTotal(List<MailboxIndexJson> mailboxesIndexInfo) {
	}

	public void setProductionStores(Map<String, ISdsSyncStore> productionStores) {
		this.productionStores = productionStores;
	}

	public void setBackupStore(SdsDataProtectSpool backupStore) {
		this.backupStore = backupStore;
	}

	public MailboxIndexJsonList runIt(String domainUid, List<String> dirEntriesUids) {
		if (dirEntriesUids.isEmpty()) {
			return new MailboxIndexJsonList(Collections.emptyList());
		}
		List<MailboxIndexJson> mailboxesIndexInfo = new ArrayList<>();
		long timeout = CoreConfig.get().getLong(CoreConfig.DataProtect.BACKUP_SDS_PART_TIME);
		Map<Integer, List<String>> partitionedDirEntriesList = dirEntriesUids.stream()
				.collect(Collectors.groupingBy(uid -> Math.abs(uid.hashCode() % PARTITIONS)));

		try (var executor = Executors.newFixedThreadPool(
				Math.min(partitionedDirEntriesList.size(), Runtime.getRuntime().availableProcessors()))) {

			var futures = partitionedDirEntriesList.entrySet().stream()
					.map(partition -> CompletableFuture
							.runAsync(() -> partTreatment(mailboxesIndexInfo, domainUid, partition), executor))
					.toList();

			CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).orTimeout(timeout, TimeUnit.HOURS)
					.join();
		}

		return new MailboxIndexJsonList(mailboxesIndexInfo);
	}

	private void partTreatment(List<MailboxIndexJson> mailboxesIndexInfo, String domainUid,
			Entry<Integer, List<String>> partition) {
		String zipHash = String.valueOf(partition.getKey());
		MailboxIndexTotal mailboxIndexAll = partitionTreatment(domainUid, zipHash, partition.getValue());
		if (mailboxIndexAll != null && mailboxIndexAll.mailboxesIndexInfo != null) {
			mailboxesIndexInfo.addAll(mailboxIndexAll.mailboxesIndexInfo);
		}
	}

	public MailboxIndexJsonList runIt(ItemValue<Domain> domain, List<String> values, boolean downloadEmailContent) {
		this.downloadEmailContent = downloadEmailContent;
		this.domain = domain;
		return runIt(domain.uid, values);
	}

	private boolean isFolderHasBeenRegenerated(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder) {
		if (folder.created == null || lastFolder.created() == null) {
			return false;
		}
		DateTimeFormatter fmt = CyrusSdsBackupFolder.DATE_FORMAT;
		return !Objects.equals(fmt.format(folder.created.toInstant()), fmt.format(lastFolder.created().toInstant()));
	}

	private boolean shouldBackupBeRebuild(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder) {
		boolean isFirstBackup = lastFolder == null;
		boolean isOldBackupFormat = !isFirstBackup && (lastFolder.version() <= 0
				|| lastFolder.messages().stream().anyMatch(message -> message.itemId <= 0));
		boolean itemIdHasBeenRegenerated = !isFirstBackup && isFolderHasBeenRegenerated(lastFolder, folder);

		return isFirstBackup || isOldBackupFormat || itemIdHasBeenRegenerated;
	}

	protected void generateSdsMailboxJson(Path outputPath, ItemValue<Domain> domain, String mailboxUid,
			String userLogin, Mailbox mailbox, List<ItemValue<MailboxFolder>> folders, ISdsSyncStore productionStore,
			boolean downloadEmailContent) throws Exception {

		Path outputTempPath = outputPath.resolveSibling(outputPath.getFileName().toString().replace(".json", ".tmp"));

		try (OutputStream outStream = new BufferedOutputStream(
				Files.newOutputStream(outputTempPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));
				JsonGenerator generator = new JsonFactory().createGenerator(outStream, JsonEncoding.UTF8)) {
			Map<String, CyrusSdsBackupFolder> lastFoldersByUid = readFromLastBackup(mailboxUid);

			generator.writeStartObject();
			generator.writeNumberField("version", 2);
			generator.writeStringField("kind", mailbox.type.name());
			generator.writeStringField("mailboxUid", mailboxUid);
			generator.writeStringField("login", userLogin);
			generator.writeStringField("domainUid", domain.uid);
			generator.writeStringField("domainName", domain.value.defaultAlias);
			generator.writeStringField("dataLocation", mailbox.dataLocation);

			generator.writeObjectFieldStart("backingstore");
			generator.writeStringField("archivekind", sysconf.stringValue(SysConfKeys.archive_kind.name()));
			generator.writeStringField("bucket", sysconf.stringValue(SysConfKeys.sds_s3_bucket.name()));
			generator.writeStringField("region", sysconf.stringValue(SysConfKeys.sds_s3_region.name()));
			generator.writeStringField("endpoint", sysconf.stringValue(SysConfKeys.sds_s3_endpoint.name()));
			generator.writeStringField("insecure", sysconf.stringValue(SysConfKeys.sds_s3_insecure.name()));
			generator.writeStringField("useSplitPath", sysconf.stringValue(SysConfKeys.sds_s3_split_path.name()));
			generator.writeEndObject();

			generator.writeArrayFieldStart("folders");
			for (ItemValue<MailboxFolder> folder : folders) {
				generateFolderEntryJson(userLogin, productionStore, downloadEmailContent, generator, lastFoldersByUid,
						folder);
			}
			generator.writeEndArray();
			generator.writeEndObject();

			moveFile(outputPath, outputTempPath);
		} catch (IOException e1) {
			logError(e1, "Unable to write {} for mailbox {}", outputTempPath, mailboxUid);
		}
	}

	private void generateFolderEntryJson(String userLogin, ISdsSyncStore productionStore, boolean downloadEmailContent,
			JsonGenerator generator, Map<String, CyrusSdsBackupFolder> lastFoldersByUid,
			ItemValue<MailboxFolder> folder) throws Exception {
		IDbMailboxRecords recordsApi = null;
		try {
			recordsApi = serviceProvider.instance(IDbMailboxRecords.class, folder.uid);
			if (recordsApi != null) {
				generator.writeStartObject();
				generator.writeStringField("uid", folder.uid);
				generator.writeStringField("fullName", folder.value.fullName);
				generator.writeStringField("name", folder.value.name);
				CyrusSdsBackupFolder lastFolder = lastFoldersByUid.get(folder.uid);
				generateSdsFolderContent(lastFolder, folder, generator, productionStore, recordsApi,
						downloadEmailContent);
				generator.writeEndObject();
			} else {
				throw new ServerFault("", ErrorCode.NOT_FOUND);
			}
		} catch (Exception ex) {
			if (ex instanceof ServerFault sf && ErrorCode.NOT_FOUND.equals(sf.getCode())) {
				logError(ex, "Unable to backup user {} folder {}: folder uid={} not found", userLogin,
						folder.value.fullName, folder.uid);
			} else {
				throw ex;
			}
		}
	}

	private Map<String, CyrusSdsBackupFolder> readFromLastBackup(String mailboxUid) {
		Map<String, CyrusSdsBackupFolder> lastFoldersByUid = new HashMap<>();

		try {
			if (lastBackup != null) {
				CyrusSdsBackupMailbox lastMailbox = lastBackup.getMailboxFromPreviousBackup(mailboxUid);
				if (lastMailbox != null) {
					lastFoldersByUid = lastMailbox.getFolders().stream()
							.collect(Collectors.toMap(f -> f.uid(), Function.identity()));
				}
			}
		} catch (Exception e) {
			logError(e, "Not able to retrieve previous backup mailbox state for uid {}", mailboxUid);
		}
		return lastFoldersByUid;
	}

	private void moveFile(Path outputPath, Path outputTempPath) {
		try {
			if (Files.exists(outputTempPath)) {
				Files.move(outputTempPath, outputPath, StandardCopyOption.REPLACE_EXISTING);
			}
		} catch (IOException e) {
			logError(e, "Unable to move {} to {}", outputTempPath, outputPath);
		}
	}

	private void generateSdsFolderContent(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder,
			JsonGenerator generator, ISdsSyncStore productionStore, IDbMailboxRecords recordsApi,
			boolean downloadEmailContent) throws IOException {
		long changesetVersion = 0;
		Stream<CyrusSdsBackupMessage> lastBackupMessages = Stream.empty();
		if (!shouldBackupBeRebuild(lastFolder, folder)) {
			changesetVersion = lastFolder.version();
			lastBackupMessages = lastFolder.messages().stream();
		}

		ContainerChangeset<Long> changeSet = recordsApi.changesetById(changesetVersion);

		Stream<CyrusSdsBackupMessage> createdMessages = getCreatedMessages(recordsApi, changeSet);
		Stream<CyrusSdsBackupMessage> backupMessage = Stream.concat( //
				removeDeletedMessage(lastBackupMessages, changeSet), //
				createdMessages //
		);

		generator.writeStringField("created", CyrusSdsBackupFolder.DATE_FORMAT.format(folder.created.toInstant()));
		generator.writeNumberField("changeSetVersion", changeSet.version);
		generator.writeArrayFieldStart("messages");

		backupMessage.forEach((CyrusSdsBackupMessage field) -> {
			// This is just a safety against broken databases, not encountered in real life
			if (field.guid != null && !field.guid.isEmpty()) {
				generateJson(generator, field);
				if (downloadEmailContent) {
					cliExportExtraAction(productionStore, field.guid);
				}
			}
		});
		generator.writeEndArray();
	}

	private Stream<CyrusSdsBackupMessage> getCreatedMessages(IDbMailboxRecords recordsApi,
			ContainerChangeset<Long> changeSet) {
		return Lists.partition(changeSet.created, 1000).stream() //
				.map(recordsApi::slice) //
				.flatMap(Collection::stream).map(irecord -> new CyrusSdsBackupMessage(irecord.value.messageBody,
						irecord.value.internalDate, irecord.itemId));
	}

	private Stream<CyrusSdsBackupMessage> removeDeletedMessage(Stream<CyrusSdsBackupMessage> lastBackupMessages,
			ContainerChangeset<Long> changeSet) {
		HashSet<Long> deletedItemsIds = new HashSet<>(changeSet.deleted);
		return lastBackupMessages.filter(msg -> !deletedItemsIds.contains(msg.itemId));
	}

	private void generateJson(JsonGenerator generator, CyrusSdsBackupMessage field) {
		try {
			generator.writeStartObject();
			generator.writeStringField("g", field.guid);
			generator.writeStringField("d", CyrusSdsBackupFolder.DATE_FORMAT.format(field.date.toInstant()));
			generator.writeNumberField("i", field.itemId);
			generator.writeEndObject();
		} catch (IOException ie) {
			ctx.error(ie, "Unable to generate json data for message_body_guid {}", field.guid);
		}
	}

	private void cliExportExtraAction(ISdsSyncStore productionStore, String guid) {
		if (productionStore != null && backupStore != null) {
			Path fp = backupStore.livePath(guid);
			if (!fp.getParent().toFile().exists()) {
				try {
					Files.createDirectories(fp.getParent());
				} catch (IOException ie) {
					throw new ServerFault(
							"Unable to create directory " + fp.getParent() + ", error message: " + ie.getMessage());
				}
			}
			SdsResponse response = productionStore.downloadRaw(GetRequest.of("", guid, fp.toString()));
			if (!response.succeeded()) {
				ctx.warn(String.format("unable to download guid %s: %s", guid, response));
			}
		}
	}
}
