/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.utils.endpoint;

import java.util.Collections;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import net.bluemind.config.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.SysConfKeys;

public class KeycloakEndpoints {
	private static final String PROTO = "/protocol/openid-connect";

	// Cannot be instantiate
	private KeycloakEndpoints() {
	}

	private static final Supplier<String> externalUrl = () -> Shared.mapSysconf().get(SysConfKeys.external_url.name());

	public static final Supplier<String> baseURL = () -> {
		ItemValue<Server> keycloakServer = Topology.get().any(TagDescriptor.bm_keycloak.getTag());
		return "http://"
				+ (DataLocation.current().equals(keycloakServer.uid) ? "127.0.0.1" : keycloakServer.value.address())
				+ ":8099/keycloak";
	};

	private static String realmsEndpoint() {
		return baseURL.get() + "/realms";
	}

	public static String tokenEndpoint(String realmId) {
		return issuerEndpoint(realmId) + PROTO + "/token";
	}

	public static String jkwsUriEndpoint(String realmId) {
		return realmsEndpoint() + "/" + realmId + PROTO + "/certs";
	}

	public static String authorizationEndpoint(String realmId) {
		return issuerEndpoint(realmId) + PROTO + "/auth";
	}

	public static String endSessionEndpoint(String realmId) {
		return issuerEndpoint(realmId) + PROTO + "/logout";
	}

	private static String issuerEndpoint(String realmId) {
		return "https://" + externalUrl.get() + "/keycloak/realms/" + realmId;
	}

	public static String masterTokenEndpoint() {
		return realmsEndpoint() + "/master/protocol/openid-connect/token";
	}

	public static String realmsAdminEndpoint() {
		return baseURL.get() + "/admin/realms";
	}

	public static String realmAdminEndpoint(String realmId) {
		return realmsAdminEndpoint() + "/" + realmId;
	}

	public static String clientsEndpoint(String realmId) {
		return realmAdminEndpoint(realmId) + "/clients";
	}

	public static String clientEndpoint(String realmId, String clientId) {
		return clientsEndpoint(realmId) + "/" + clientId;
	}

	public static String clientCredsEndpoint(String realmId, String clientId) {
		return clientEndpoint(realmId, clientId) + "/client-secret";
	}

	public static String componentsEndpoint(String realmId) {
		return realmAdminEndpoint(realmId) + "/components";
	}

	public static String componentEndpoint(String realmId, String componentId) {
		return componentsEndpoint(realmId) + "/" + componentId;
	}

	private static String authenticationEndpoint(String realmId) {
		return realmAdminEndpoint(realmId) + "/authentication";
	}

	public static String flowsEndpoint(String realmId) {
		return authenticationEndpoint(realmId) + "/flows";
	}

	public static String flowEndpoint(String realmId, String flowId) {
		return flowsEndpoint(realmId) + "/" + flowId;
	}

	public static String copyFlowEndpoint(String realmId, String flowId) {
		return flowEndpoint(realmId, flowId) + "/copy";
	}

	public static String flowExecutionsEndpoint(String realmId, String flowId) {
		return flowEndpoint(realmId, flowId) + "/executions";
	}

	public static String authenticationExecutionsEndpoint(String realmId) {
		return authenticationEndpoint(realmId) + "/executions";
	}

	public static String authenticationExecutionEndpoint(String realmId, String executionId) {
		return authenticationExecutionsEndpoint(realmId) + "/" + executionId;
	}

	public static String authenticationExecutionRaisePriorityEndpoint(String realmId, String executionId) {
		return authenticationExecutionEndpoint(realmId, executionId) + "/raise-priority";
	}

	public static String authenticationExecutionConfigEndpoint(String realmId, String executionId) {
		return authenticationExecutionEndpoint(realmId, executionId) + "/config";
	}

	private static String sessionsEndpoint(String realmId) {
		return realmAdminEndpoint(realmId) + "/sessions";
	}

	public static String sessionEndpoint(String realmId, String sessionId) {
		return sessionEndpoint(realmId, sessionId, Collections.emptyMap());
	}

	public static String sessionEndpoint(String realmId, String sessionId, Map<String, String> parameters) {
		String parametersAsString = parameters.entrySet().stream().map(e -> e.getKey() + "=" + e.getValue())
				.collect(Collectors.joining("&"));

		String sessionEndpoint = sessionsEndpoint(realmId) + "/" + sessionId;
		if (parametersAsString == null || parametersAsString.isBlank()) {
			return sessionEndpoint;
		}

		return sessionEndpoint + "?" + parametersAsString;
	}
}
