/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.dataprotect.resource.impl.pg;

import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.calendar.api.ICalendarsMgmt;
import net.bluemind.calendar.api.IVEvent;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.email.SendIcs;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;
import net.bluemind.resource.api.IResources;
import net.bluemind.resource.api.ResourceDescriptor;

public class SendResourceICSTaskPg {

	private static final Logger logger = LoggerFactory.getLogger(SendResourceICSTaskPg.class);

	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;
	private final RestoreActionExecutor<EmailData> executor;
	private final ResourceBundle bundle;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	@SuppressWarnings("unchecked")
	public SendResourceICSTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item,
			RestoreActionExecutor<? extends IRestoreActionData> executor, ResourceBundle bundle) {
		this.backup = backup;
		this.restorableItem = item;
		this.executor = (RestoreActionExecutor<EmailData>) executor;
		this.bundle = bundle;
	}

	public void run() throws Exception {
		logger.info("Starting export for uid {}", restorableItem.entryUid());
		SendIcs sendMail = new SendIcs(restorableItem, executor);

		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());

			IResources ouBackup = back.provider().instance(IResources.class, restorableItem.item().domainUid);
			ItemValue<ResourceDescriptor> backupOu = ouBackup.getComplete(restorableItem.entryUid());

			String resourceCal = ICalendarUids.resourceCalendar(backupOu.uid);
			CalendarDescriptor calendarDescriptor = back.provider().instance(ICalendarsMgmt.class).get(resourceCal);

			IVEvent service = back.provider().instance(IVEvent.class, resourceCal);
			Map<String, String> allIcs = Map.of(calendarDescriptor.name,
					GenericStream.streamToString(service.exportAll()));

			restorableItem.monitorLog("Export '{}' events from {} [uid={}]", allIcs.size(),
					restorableItem.displayName(), restorableItem.entryUid());

			sendMail.sendMessage(allIcs, bundle.getString("send.resource.restore.message"),
					bundle.getString("send.resource.restore.subject"));
		} catch (Exception e) {
			logger.error("Error while sending resource calendars", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.RESOURCE));
		} finally {
			restorableItem.endTask();
		}
	}

}
