/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2017
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.user.accounts.service.internal;

import java.sql.SQLException;
import java.util.Base64;
import java.util.List;
import java.util.stream.Collectors;

import net.bluemind.core.annotationvalidator.AnnotationValidator;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.validator.IValidatorFactory;
import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.system.service.ExternalSystemsRegistry;
import net.bluemind.system.service.RegisteredExternalSystem;
import net.bluemind.user.accounts.hook.IUserAccountsHook;
import net.bluemind.user.accounts.persistence.UserAccountsStore;
import net.bluemind.user.api.IInternalUserExternalAccount;
import net.bluemind.user.api.UserAccount;
import net.bluemind.user.api.UserAccountInfo;

public class UserAccountsService implements IInternalUserExternalAccount {

	private RBACManager rbacManager;
	private final UserAccountsStore store;
	private final String domainUid;
	private final Item user;
	private IValidatorFactory<UserAccount> validator;
	private BmContext context;

	private static List<IUserAccountsHook> hooks = getHooks();

	public UserAccountsService(BmContext context, Container domain, Item user) {
		this.rbacManager = new RBACManager(context).forDomain(domain.uid);
		this.store = new UserAccountsStore(context.getDataSource());
		this.domainUid = domain.uid;
		this.user = user;
		validator = new AnnotationValidator.GenericValidatorFactory<>(UserAccount.class);
		this.context = context;
	}

	private static List<IUserAccountsHook> getHooks() {
		RunnableExtensionLoader<IUserAccountsHook> loader = new RunnableExtensionLoader<>();
		return loader.loadExtensions("net.bluemind.user.accounts", "userAccountsHook", "hook", "class");
	}

	@Override
	public void create(String systemIdentifier, UserAccount account) throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);
		validator.create(context);

		RegisteredExternalSystem system = ExternalSystemsRegistry.getSystem(systemIdentifier);
		UserAccount sanitizedAccount = null != system ? system.sanitizeAccount(domainUid, account) : account;

		if (null != sanitizedAccount.credentials) {
			b64Credentials(sanitizedAccount);
		}
		JdbcAbstractStore.doOrFail(() -> {
			store.create(user, systemIdentifier, sanitizedAccount);
			return null;
		});

		hooks.forEach(hook -> hook.onCreate(domainUid, user.uid, systemIdentifier, sanitizedAccount));
	}

	@Override
	public void update(String systemIdentifier, UserAccount account) throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);
		validator.create(context);

		RegisteredExternalSystem system = ExternalSystemsRegistry.getSystem(systemIdentifier);
		UserAccount sanitizedAccount = null != system ? system.sanitizeAccount(domainUid, account) : account;

		JdbcAbstractStore.doOrFail(() -> {
			if (null != sanitizedAccount.credentials) {
				b64Credentials(sanitizedAccount);
			} else {
				sanitizedAccount.credentials = store.get(user, systemIdentifier).credentials;
			}
			store.update(user, systemIdentifier, sanitizedAccount);
			return null;
		});

		hooks.forEach(hook -> hook.onUpdate(domainUid, user.uid, systemIdentifier, account));
	}

	@Override
	public UserAccount get(String systemIdentifier) throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);

		return JdbcAbstractStore.doOrFail(() -> decorate(store.get(user, systemIdentifier)));
	}

	@Override
	public List<UserAccountInfo> getAll() throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);

		return JdbcAbstractStore.doOrFail(() -> store.getAll(user).stream()//
				.map(i -> new UserAccountInfo(decorate(i), i.externalSystemId)) //
				.collect(Collectors.toList()));
	}

	@Override
	public void delete(String systemIdentifier) throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);

		JdbcAbstractStore.doOrFail(() -> {
			store.delete(user, systemIdentifier);
			return null;
		});

		hooks.forEach(hook -> hook.onDelete(domainUid, user.uid, systemIdentifier));
	}

	@Override
	public void deleteAll() throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);

		List<UserAccountInfo> infos = getAll();
		JdbcAbstractStore.doOrFail(() -> {
			store.deleteAll(user);
			return null;
		});

		hooks.forEach(hook -> infos//
				.forEach(info -> hook.onDelete(domainUid, user.uid, info.externalSystemId)));
	}

	@Override
	public String getCredentials(String systemIdentifier) throws ServerFault {
		rbacManager.forEntry(user.uid).check(BasicRoles.ROLE_MANAGE_EXTERNAL_ACCOUNTS);

		return JdbcAbstractStore.doOrFail(() -> deB64Credentials(store.get(user, systemIdentifier).credentials));
	}

	private <T extends UserAccount> T decorate(T account) {
		if (null == account) {
			return null;
		}
		account.credentials = null;
		return account;
	}

	private String deB64Credentials(String credentials) {
		return new String(Base64.getDecoder().decode(credentials));
	}

	private void b64Credentials(UserAccount account) {
		account.credentials = Base64.getEncoder().encodeToString(account.credentials.getBytes());
	}

	@Override
	public void restore(ItemValue<UserAccount> item, boolean isCreate) {
		if (isCreate) {
			create(item.uid, item.value);
		} else {
			update(item.uid, item.value);
		}
	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		try {
			return new ItemValueExists(false, store.get(user, uid) != null);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
