/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.config;

import java.util.Map;

import net.bluemind.domain.service.DomainUrlValidator;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.hook.ISystemConfigurationSanitizor;
import net.bluemind.system.hook.ISystemConfigurationValidator;

public class ExternalUrlHook implements ISystemConfigurationSanitizor, ISystemConfigurationValidator {
	@Override
	public void sanitize(SystemConf previous, Map<String, String> modifications) {
		if (!modifications.containsKey(SysConfKeys.external_url.name()) && previous != null
				&& previous.values.containsKey("external-url")) {
			modifications.put(SysConfKeys.external_url.name(), previous.stringValue("external-url"));
		}
		// external-url and other-urls keys are forbidden in database
		modifications.put("external-url", null);
		modifications.put("other-urls", null);

		String url = modifications.get(SysConfKeys.external_url.name());
		if (url != null) {
			modifications.put(SysConfKeys.external_url.name(), url.trim());
		}

		url = modifications.get(SysConfKeys.other_urls.name());
		if (url != null) {
			modifications.put(SysConfKeys.other_urls.name(), url.trim().replace(" +", " "));
		}
	}

	@Override
	public void validate(SystemConf previous, Map<String, String> modifications) {
		if (!modifications.containsKey(SysConfKeys.external_url.name())
				&& !modifications.containsKey(SysConfKeys.other_urls.name())) {
			return;
		}

		String newUrl = getUrl(SysConfKeys.external_url, previous, modifications);
		String newOtherUrls = getUrl(SysConfKeys.other_urls, previous, modifications);

		DomainUrlValidator.forGlobalUrl(newUrl,
				previous == null ? null : previous.stringValue(SysConfKeys.external_url.name()), newOtherUrls,
				previous == null ? null : previous.stringValue(SysConfKeys.other_urls.name()));
	}

	private String getUrl(SysConfKeys url, SystemConf previous, Map<String, String> modifications) {
		if (modifications.containsKey(url.name())) {
			return modifications.get(url.name());
		}

		if (previous == null) {
			return null;
		}

		return previous.stringValue(url.name());
	}
}
